// SQuadTree.cpp: implementation of the SQuadTree class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Simulator.h"
#include "SQuadTree.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

SQuadTree::SQuadTree()
{
    isOnSim = false;
    DirectXObjectList = NULL;

}

SQuadTree::~SQuadTree()
{

}

/************************************************************************
* Function FPoint DetectCollision(FPoint ObjectStart, FPoint ObjectEnd, FPoint Delta)  
*
* PURPOSE
* This function detects a colision between a moving line (model) and an static object (quadtree)
* INPUTS
* ObjectStart, the start of the line which is moving
* ObjectEnd, the end of the line which is moving
* Delta, the displacment of the line
* USAGE
* By the model class to see if valid movement can occur
* ALOGORITHM
* 1) Since the collisions happen at a point, generate a points list
*    a) Either a line starts/ends in the collision path, this is a valid point
*    b) Or the line intersects with the collision bound path which is then a valid point
* 2) Check if the points are in the collision bounds, by checking if it in that area rect
* 3) Take the closest point
* RETURNS
* If There is a colosion it returns the maximum delta that can be achieved without a colision
* Otherwise returns -1000000 (both x and y) which will never happen.
*************************************************************************/
FPoint SQuadTree::DetectCollision(FPoint ObjectStart, FPoint ObjectEnd, FPoint DeltaStart, FPoint DeltaEnd)
{
    // get the area of interest
    CRect Area = DetectCollisionFormRect(ObjectStart,ObjectEnd,DeltaStart,DeltaEnd);
    FRect AreaF = DetectCollisionFormRectF(ObjectStart,ObjectEnd,DeltaStart,DeltaEnd);

    // create the four bounding lines
    ULinearAlgebra::Matrix2X2 StartLine = LAUtils.PointstoLine(ObjectStart,ObjectEnd);
    ULinearAlgebra::Matrix2X2 DeltaLine = LAUtils.PointstoLine(DeltaStart,DeltaEnd);
    ULinearAlgebra::Matrix2X2 PerpLine1 = LAUtils.PointstoLine(ObjectStart,DeltaStart);
    ULinearAlgebra::Matrix2X2 PerpLine2 = LAUtils.PointstoLine(ObjectEnd,DeltaEnd);

    PointListPtr TheList = NULL;
    PointListPtr tempo;

    
    // crappy semiphore because this function is reused for a few things which uses
    // a single enum pointer.
    if (!isOnSim)
    {
        isOnSim = true;

        // create a list with all objects that possibily occur in that area
        SimCreateList(Area);

        ObjectPtr object = NULL;

        // Start looking at those objects.
        while (SimEnumObjects(object))
        {
            // don't want an arc object, not supported
            if (object->objecttype == LINE)
            {
                CRect Area1;
                FPoint tPoint;             

                Area1 = DetectCollisionFormRect(object->start,object->end);

                // add both the start and end points to the points lists
                tempo = new PointList;
                tempo->Point.x = (float)object->start.x;
                tempo->Point.y = (float)object->start.y;
                tempo->Next = TheList;
                TheList = tempo;

                tempo = new PointList;
                tempo->Point.x = (float)object->end.x;
                tempo->Point.y = (float)object->end.y;
                tempo->Next = TheList;
                TheList = tempo;

                // Add the intersections
                ULinearAlgebra::Matrix2X2 TempLine = LAUtils.PointstoLine(object->start,object->end);

                tPoint = LAUtils.FindIntersectionF(PerpLine1,TempLine);
                if (LAUtils.InRect(tPoint,Area1))
                {
                    tempo = new PointList;
                    tempo->Point = tPoint;
                    tempo->Next = TheList;
                    TheList = tempo;
                }

                tPoint = LAUtils.FindIntersectionF(PerpLine2,TempLine);
                if (LAUtils.InRect(tPoint,Area1))
                {
                    tempo = new PointList;
                    tempo->Point = tPoint;
                    tempo->Next = TheList;
                    TheList = tempo;
                }  
            }

        }

        isOnSim=false;
    }

    FPoint Collision;
    Collision.x = NOCOLLISION;
    Collision.y = NOCOLLISION;

    float MinPoint = (float)99999999.0;

    // ok now looks at the points list
    while (TheList != NULL)
    {
        // two stage check if valid point
        // this one checks if it is in the general bounds, of the new displacements.  The old one shouldn't be looked at.
        FRect NewArea;

        // this is for the special case of a line traveling on the same slope as itself (e.g the 90 degree line)
        // an error happens if it tries to back away from a perpindicular (180) collision it detects another collision
        // because of a 90 degree special case in the IsBetweenLines function call below.  This
        // seems to fix it without causing any other side effects.
        if ((StartLine.A == DeltaLine.A) && (StartLine.B == DeltaLine.B)
            && (StartLine.C == DeltaLine.C))
            NewArea = DetectCollisionFormRectF(DeltaStart,DeltaEnd,DeltaStart,DeltaEnd);
        else
            NewArea = AreaF;

        if (LAUtils.InRect(TheList->Point,NewArea))         
        {
            // this one checks if it is in the delta area (line check so doesn't end at the end points
            // see above check
            if (LAUtils.IsBetweenLines(StartLine,DeltaLine,TheList->Point))
            {
                float ADistance = LAUtils.GetDistance(StartLine,TheList->Point);
                ADistance = LAUtils.Abs(ADistance);
                if (ADistance < MinPoint)
                {
                    MinPoint = ADistance;
                    Collision.x = TheList->Point.x;
                    Collision.y = TheList->Point.y;
                }
            }
        }
        // traverse and clean up
        tempo = TheList;
        TheList = TheList->Next;
        delete tempo;
        tempo = NULL;
    }   


    return Collision;


}

/************************************************************************
* Function CRect DetectCollisionFormRect(FPoint ObjectStart, FPoint ObjectEnd, FPoint Delta) 
*
* PURPOSE
* To return a rectange which contains all of the points in between the line and its delta
* INPUTS
* ObjectStart, the start of the line which is moving
* ObjectEnd, the end of the line which is moving
* DeltaStart, the start + displacement of the line
* DeltaEnd, the end + displacement of the line 
* RETURNS
* The rect (in real coordinates) of the area incompassed
**************************************************************************/
CRect SQuadTree::DetectCollisionFormRect(FPoint ObjectStart, FPoint ObjectEnd, FPoint DeltaStart, FPoint DeltaEnd)
{
   // FPoint DeltaStart, DeltaEnd;

    CRect toReturn;

    // find the top
    toReturn.top = (long)ObjectStart.y;
    if (ObjectEnd.y < toReturn.top)
        toReturn.top = (long)ObjectEnd.y;
    if (DeltaStart.y < toReturn.top)
        toReturn.top = (long)DeltaStart.y;
    if (DeltaEnd.y < toReturn.top)
        toReturn.top = (long)DeltaEnd.y;

    // find the left
    toReturn.left = (long)ObjectStart.x;
    if (ObjectEnd.x < toReturn.left)
        toReturn.left = (long)ObjectEnd.x;
    if (DeltaStart.x < toReturn.left)
        toReturn.left = (long)DeltaStart.x;
    if (DeltaEnd.x < toReturn.left)
        toReturn.left = (long)DeltaEnd.x;

    // find the bottom
    toReturn.bottom = (long)ObjectStart.y;
    if (ObjectEnd.y > toReturn.bottom)
        toReturn.bottom = (long)ObjectEnd.y;
    if (DeltaStart.y > toReturn.bottom)
        toReturn.bottom = (long)DeltaStart.y;
    if (DeltaEnd.y > toReturn.bottom)
        toReturn.bottom = (long)DeltaEnd.y;

    // find the right
    toReturn.right = (long)ObjectStart.x;
    if (ObjectEnd.x > toReturn.right)
        toReturn.right = (long)ObjectEnd.x;
    if (DeltaStart.x > toReturn.right)
        toReturn.right = (long)DeltaStart.x;
    if (DeltaEnd.x > toReturn.right)
        toReturn.right = (long)DeltaEnd.x;

    return toReturn;

}

/************************************************************************
* Function CRect DetectCollisionFormRect(FPoint ObjectStart, FPoint ObjectEnd, FPoint Delta) 
*
* PURPOSE
* To return a rectange which contains all of the points in between the line and its delta
* INPUTS
* ObjectStart, the start of the line which is moving
* ObjectEnd, the end of the line which is moving
* DeltaStart, the start + displacement of the line
* DeltaEnd, the end + displacement of the line 
* RETURNS
* The rect (in real coordinates) of the area incompassed
**************************************************************************/
FRect SQuadTree::DetectCollisionFormRectF(FPoint ObjectStart, FPoint ObjectEnd, FPoint DeltaStart, FPoint DeltaEnd)
{
   // FPoint DeltaStart, DeltaEnd;

    FRect toReturn;

    // find the top
    toReturn.top = ObjectStart.y;
    if (ObjectEnd.y < toReturn.top)
        toReturn.top = ObjectEnd.y;
    if (DeltaStart.y < toReturn.top)
        toReturn.top = DeltaStart.y;
    if (DeltaEnd.y < toReturn.top)
        toReturn.top = DeltaEnd.y;

    // find the left
    toReturn.left = ObjectStart.x;
    if (ObjectEnd.x < toReturn.left)
        toReturn.left = ObjectEnd.x;
    if (DeltaStart.x < toReturn.left)
        toReturn.left = DeltaStart.x;
    if (DeltaEnd.x < toReturn.left)
        toReturn.left = DeltaEnd.x;

    // find the bottom
    toReturn.bottom = ObjectStart.y;
    if (ObjectEnd.y > toReturn.bottom)
        toReturn.bottom = ObjectEnd.y;
    if (DeltaStart.y > toReturn.bottom)
        toReturn.bottom = DeltaStart.y;
    if (DeltaEnd.y > toReturn.bottom)
        toReturn.bottom = DeltaEnd.y;

    // find the right
    toReturn.right = ObjectStart.x;
    if (ObjectEnd.x > toReturn.right)
        toReturn.right = ObjectEnd.x;
    if (DeltaStart.x > toReturn.right)
        toReturn.right = DeltaStart.x;
    if (DeltaEnd.x > toReturn.right)
        toReturn.right = DeltaEnd.x;

    return toReturn;

}

/************************************************************************
* Function CRect DetectCollisionFormRect(CPoint Start, CPoint End)
*
* PURPOSE
* To return a rectange which contains all of the points in a line
* INPUTS
* Start, the start of the line which is moving
* End, the end of the line which is moving
* RETURNS
* The rect (in real coordinates) of the area incompassed
**************************************************************************/
CRect SQuadTree::DetectCollisionFormRect(CPoint Start, CPoint End)
{
    CRect toReturn;

    // find the top
    toReturn.top = (long)Start.y;
    if (End.y < toReturn.top)
        toReturn.top = (long)End.y;
    
    // find the left
    toReturn.left = (long)Start.x;
    if (End.x < toReturn.left)
        toReturn.left = (long)End.x;
    
    // find the bottom
    toReturn.bottom = (long)Start.y;
    if (End.y > toReturn.bottom)
        toReturn.bottom = (long)End.y;
   
    // find the right
    toReturn.right = (long)Start.x;
    if (End.x > toReturn.right)
        toReturn.right = (long)End.x;
   
    return toReturn;

}

/************************************************************************
* Function CRect DetectCollisionFormRect(CPoint Start, CPoint End)
*
* PURPOSE
* To return a rectange which contains all of the points in a line
* INPUTS
* Start, the start of the line which is moving
* End, the end of the line which is moving
* RETURNS
* The rect (in real coordinates) of the area incompassed
**************************************************************************/
FRect SQuadTree::DetectCollisionFormRect(FPoint Start, FPoint End)
{
    FRect toReturn;

    // find the top
    toReturn.top = Start.y;
    if (End.y < toReturn.top)
        toReturn.top = End.y;
    
    // find the left
    toReturn.left = Start.x;
    if (End.x < toReturn.left)
        toReturn.left = End.x;
    
    // find the bottom
    toReturn.bottom = Start.y;
    if (End.y > toReturn.bottom)
        toReturn.bottom = End.y;
   
    // find the right
    toReturn.right = Start.x;
    if (End.x > toReturn.right)
        toReturn.right = End.x;
   
    return toReturn;

}

/************************************************************************
* Function float DetectCollisionDistance(FPoint ThePoint, FPoint CollisionPoint, FPoint ObjectStart, FPoint ObjectEnd, FPoint DeltaStart, FPoint DeltaEnd)
*
* PURPOSE
* To return the perpindical distances of a point of a line which colides with a line, which
* is specified by collisionPoint
* USAGE
* Used to move the model up to the collision point.  There was one snag with the algorithm
* Between choosing points and this function was needed to resolve it.
* RETURNS
* The perpindicular distance between a point and a line.  -1 if not valid.
**************************************************************************/
float SQuadTree::DetectCollisionDistance(FPoint ThePoint, FPoint CollisionPoint, FPoint ObjectStart, FPoint ObjectEnd, FPoint DeltaStart, FPoint DeltaEnd)
{

    float tReturn = -1;
    
    // get the area of interest
    CRect Area = DetectCollisionFormRect(ObjectStart,ObjectEnd,DeltaStart,DeltaEnd);

    // crappy semiphore because this function is reused for a few things which uses
    // a single enum pointer.
    if (!isOnSim)
    {
        isOnSim = true;

        // create a list with all objects that possibily occur in that area
        SimCreateList(Area);

        ObjectPtr object = NULL;

        // Start looking at those objects.
        while (SimEnumObjects(object))
        {
            // don't want an arc object, not supported
            if (object->objecttype == LINE)
            {
                ULinearAlgebra::Matrix2X2 Line = LAUtils.PointstoLine(object->start,object->end);
                // check if we are looking at the right line
                if (LAUtils.isPointonLine(CollisionPoint,Line))
                {
                    tReturn = LAUtils.GetDistance(object->start,object->end,ThePoint);
                }

            }            
        }
        isOnSim=false;
    }
    return tReturn;
}

/************************************************************************
* Function BOOL DetectCollisionSafetyCheck(CPoint Start, CPoint End)
*
* PURPOSE
* Despite my best attempts there at still problems with the collision detect to remedy this
* A safety check checks if the new object is currently intersecting something by accedent.  If it
* is then don't put it there.  Also could be used when first placing the model
* USAGE
* To make sure the model isn't intersecting  a wall
* RETURNS
* true or false (hopefully false);
**************************************************************************/
BOOL SQuadTree::DetectCollisionSafetyCheck(FPoint Start, FPoint End)
{
    // get the area of interest
    FRect Area = DetectCollisionFormRect(Start,End);

    ULinearAlgebra::Matrix2X2 Line = LAUtils.PointstoLine(Start,End);

    BOOL tReturn = false;
    
    // crappy semiphore because this function is reused for a few things which uses
    // a single enum pointer.
    if (!isOnSim)
    {
        isOnSim = true;

        // create a list with all objects that possibily occur in that area
        CRect Area1;
        Area1.top = (long)Area.top;
        Area1.bottom = (long)Area.bottom;
        Area1.left = (long)Area.left;
        Area1.right = (long)Area.right;
        SimCreateList(Area1);

        ObjectPtr object = NULL;

        // Start looking at those objects.
        while (SimEnumObjects(object))
        {
            // don't want an arc object, not supported
            if (object->objecttype == LINE)
            {
                ULinearAlgebra::Matrix2X2 Line2 = LAUtils.PointstoLine(object->start,object->end);
                FPoint Start,End;
                Start.x = (float)object->start.x;
                Start.y = (float)object->start.y;
                End.x = (float)object->end.x;
                End.y = (float)object->end.y;
                FRect Line2Area = DetectCollisionFormRect(Start,End);

                // find the intersection
                FPoint Intersect = LAUtils.FindIntersectionF(Line,Line2);

                // if the intersectino point is valid then we have a problem, return true
                if (LAUtils.InRect(Intersect,Area))
                {
                    // check the other line too
                    if (LAUtils.InRect(Intersect,Line2Area))
                    {
                        tReturn = true;
                    }
                }
            }
        }
        isOnSim=false;
    }
    return tReturn;
}

/************************************************************************
* Function FPoint DetectCollision(FPoint StartPoint, float Angle, long MaxRange)
*
* PURPOSE
* To check if a point extended on the angle, given a maximum range collides with anything
* USAGE
* Currently by the IR to detect if it detects something
* ALGORITHM
* 1) Get all the quadrants of interest
* 2) check if there is an intersection and if the distance is less/equal then maxrange
* 3) Add an extra thing to see if it is in the right direction
* 4) return point or -1
* RETURNS
* The collision point.  -1 if no collision within range.
**************************************************************************/
FPoint SQuadTree::DetectCollision(FPoint StartPoint, float Angle, long MaxRange)
{

    FPoint tReturn;//return variable
    tReturn.x = -1;
    float ShortestIntersection = (float)MaxRange;

    // cheap way to find end point by using existing rotate function
    FPoint EndPoint;
    EndPoint.x = StartPoint.x + MaxRange;
    EndPoint.y = StartPoint.y;

    EndPoint = LAUtils.RotatePoint(EndPoint,StartPoint,Angle);

    ULinearAlgebra::Matrix2X2 Line = LAUtils.PointstoLine(StartPoint,EndPoint); 
    
    // crappy semiphore because this function is reused for a few things which uses
    // a single enum pointer.
    if (!isOnSim)
    {
        isOnSim = true;
       
         // form the area of interest
        CRect Area = DetectCollisionFormRect(EndPoint,StartPoint,EndPoint,StartPoint);
        SimCreateList(Area);

        ObjectPtr object = NULL;

        // Start looking at those objects.
        while (SimEnumObjects(object))
        {
            // don't want an arc object, not supported
            if (object->objecttype == LINE)
            {
                ULinearAlgebra::Matrix2X2 ObjectLine = LAUtils.PointstoLine(object->start,object->end);
                FPoint Intersection = LAUtils.FindIntersectionF(ObjectLine,Line);

                BOOL WrongDirection = false;
                float CorrectAngle = LAUtils.Mod360(Angle);
                // make sure the beam is looking forware
                if ((CorrectAngle > 90) && (CorrectAngle < 270))
                {
                    if (Intersection.x > StartPoint.x)
                    {
                        // wrong direction the point is invalid
                        WrongDirection = true;
                    }
                }
                else
                {
                    if (Intersection.x < StartPoint.x)
                    {
                        // wrong direction the point is invalid
                        WrongDirection = true;
                    }
                }

                 if ((CorrectAngle > 180) && (CorrectAngle < 360))
                 {
                     if (Intersection.y < StartPoint.y)
                     {
                         //wrong direction the point is invalid
                           WrongDirection = true;
                     }
                 }
                 else
                 {
                     if (Intersection.y > StartPoint.y)
                     {
                         //wrong direction the point is invalid
                           WrongDirection = true;
                     }
                 }


                if (!WrongDirection)
                {
                    // make sure it is on the line
                    CRect LineBounds = DetectCollisionFormRect(object->start,object->end);
                    if (LAUtils.InRect(Intersection,LineBounds))
                    {
                        float Distance = LAUtils.GetDistance(Intersection,StartPoint);
                        if (Distance < ShortestIntersection)
                        {
                            ShortestIntersection = Distance;
                            tReturn = Intersection;
                        }
                    }
                }
            }
        }
        isOnSim=false;
    }
    else
    {
        int adsf = 234;
        
    }

    return tReturn;
}


/************************************************************************
* SimCreateList(CRect SimBound)
*
* PURPOSE
* The non recursive call which calls the recursive one (same name).  It initializes 
* the Simulation list, and calls the recursive function
* USAGE
* By the simulation to get a list of objects in an area
* INPUT
* CRect SimBound, the area which the object are in
*
*************************************************************************/
void SQuadTree::SimCreateList(CRect SimBound)
{
       // now find the quadrants that should be loooked at.
    CRect tSimBound;
    tSimBound.left = (SimBound.left-1)/100+1;
    tSimBound.top = (SimBound.top-1)/100+1;
    tSimBound.right = (SimBound.right-1)/100+1;
    tSimBound.bottom = (SimBound.bottom-1)/100+1;

    // make sure all quadrants are in bounds (1-128)
    if (tSimBound.left < QUADXSTART)
        tSimBound.left = QUADXSTART;
    if (tSimBound.right > QUADXBOUND)
        tSimBound.right = QUADXBOUND;
    if (tSimBound.top < QUADYSTART)
        tSimBound.top = QUADYSTART;
    if (tSimBound.bottom > QUADYBOUND)
        tSimBound.bottom = QUADYBOUND;

    SimObjectListPtr = NULL; // just in case

    // increment the sim cycle
    SimCycle++; 

    SimCreateList(tSimBound,HeadQuadPtr);


}

/************************************************************************
* SimCreateList(CRect SimBound, TreePtr toSim)
*
* PURPOSE
* The call which does a decent of the quad tree, which finds the minimum number
* of nodes that the simulator needs 
* USAGE
* SimCreateList (non recursive version)
* INPUT
* CRect SimBound, the area which the object need to be looked at
* TREESTRUCT *toSim the node in the tree which is being looked at.
*************************************************************************/
void SQuadTree::SimCreateList(CRect SimBound, TreePtr toSim)
{
    // check if current node, is completely contained in the simulation bounds
    if ((toSim->QuadRect.bottom <= SimBound.bottom)
        && (toSim->QuadRect.top >= SimBound.top)
        && (toSim->QuadRect.left >= SimBound.left)
        && (toSim->QuadRect.right <= SimBound.right))
    {
        SimAddtoList(toSim);
        return;
    }
    // check if none of the current node is contained in the simulation bounds
    else if ((toSim->QuadRect.top > SimBound.bottom)
        || (toSim->QuadRect.bottom < SimBound.top)
        || (toSim->QuadRect.right < SimBound.left)
        || (toSim->QuadRect.left > SimBound.right))
    {
        return;
    }
    // O.K there is a partial match, lets get each quad of this node
    else
    {
        // note there should never be a case that we are here, and the NextQuad is NULL,
        // so this isn't taken into account.
        for (int ix = 0; ix < QUAD; ix++)
        {
            SimCreateList(SimBound,toSim->NextQuad[ix]);

        }
    }
    return;



}

/************************************************************************
* SimAddtoList(UQuadTree::TREESTRUCT *toSim)
*
* PURPOSE
* This function is used by the recursive simulator list call, to add a the objects of a node
* onto the simulator list.
* USAGE
* SimCreateList (the recursive one)
* INPUT
* TREESTRUCT *toSim, a pointer to the node which is to be looked at
*
*************************************************************************/
void SQuadTree::SimAddtoList(TreePtr toSim)
{
    // create a temporary pointer to traverse the list
    ObjectListPtr tObjectListPtr = toSim->Objects;
    ObjectListPtr tPtr;

      // there has to be some elements in the nodes list, to add them to the list
    if (tObjectListPtr != NULL)
    {
        // no elements, insert in top
        if (SimObjectListPtr == NULL)
        {
            SimObjectListPtr = new OBJECTLIST;
            // since it is the first element, I don't have to take into acount of the object
            // already has been added to this list.
            SimObjectListPtr->Data = tObjectListPtr->Data;
            SimObjectListPtr->Next = NULL; // note only using a single linked list.
            SimObjectListPtr->Data->SimFlag = SimCycle; // flag is so it isn't placed in the list again.             
            tObjectListPtr = tObjectListPtr->Next;
        }
        while (tObjectListPtr != NULL)
        {
            // check to see if it has been added to the list yet.
            if (tObjectListPtr->Data->SimFlag != SimCycle)
            {
                tPtr = new OBJECTLIST;
                tPtr->Data = tObjectListPtr->Data;
                tPtr->Data->SimFlag = SimCycle; // flag is so it isn't placed in the list again.            
                tPtr->Next = SimObjectListPtr; // note only using a single linked list.
                SimObjectListPtr = tPtr;
                
            }
            tObjectListPtr = tObjectListPtr->Next;
        }
    }

}

/************************************************************************
* Function BOOL SimEnumObjects(ObjectPtr &object)
* 
* PURPOSE
* To return an object for the simulator to look at
* USAGE
* by lot of functions in the simulator
* INPUTS
* None, it has its own pointer to keep track of an objectlist, curtesy of the 
* SimCreateList function which must be called first.
* OUTPUTS
* object, an object which is contained in the area to be looked at
* Returns true if it returns a valid object, false if the object is not valid
*************************************************************************/
BOOL SQuadTree::SimEnumObjects(ObjectPtr &object)
{

    if (SimObjectListPtr == NULL)
    {
        return false;
    }
    else
    {
        ObjectListPtr old;
        object = SimObjectListPtr->Data;
        old = SimObjectListPtr;
        SimObjectListPtr = SimObjectListPtr->Next;
        delete old;
        old = NULL;
        return true;
    }


}

/************************************************************************
* Function void DirectXEnumWallList()
* 
* PURPOSE
* To go through the wall list to get wall informations
* USAGE
* By the directX class
*************************************************************************/
BOOL SQuadTree::DirectXEnumWallList(SQuadTree::WallList &Wall)
{
    if (DirectXObjectList == NULL)
    {
        return false;
    }
    else
    {
        while ((DirectXObjectList->Data->objecttype != LINE)
            && DirectXObjectList->Data->objecttype != TRIANGLE3D)
        {
            DirectXObjectList = DirectXObjectList->Next;
            if (DirectXObjectList == NULL)
                return false;
        }       

        Wall.Start = DirectXObjectList->Data->start;
        Wall.End = DirectXObjectList->Data->end;

        if (DirectXObjectList->Data->objecttype == LINE)
            Wall.Point3.x = -1;
        else if (DirectXObjectList->Data->objecttype == TRIANGLE3D)
            Wall.Point3 = DirectXObjectList->Data->middle;

        DirectXObjectList = DirectXObjectList->Next;
    }
    return true;
}

/************************************************************************
* Function DirectXCreateWallList()
* 
* PURPOSE
* Creates an object list interally for directX rendering
* Currently it just looks at the bigobjectlist
*************************************************************************/
void SQuadTree::DirectXCreateWallList()
{
    DirectXObjectList = BigObjectList;

}
