// Box.cpp: implementation of the Box class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "simulator.h"
#include "Box.h"
#include "math.h"
#include "VTManager.h"
#include "LightMaps.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Box::Box()
{

}

Box::~Box()
{

}

//////////////////////////////////////////////////////////////////////
// SetParameters(double AngleX, double AngleY, double AngleZ, double HalfX, double HalfY, double HalfZ, Point3 Center)
//
// Sets the parameters of the box and computes its bounding volumes
//////////////////////////////////////////////////////////////////////
void Box::SetParameters(double AngleX, double AngleY, double AngleZ, double HalfX, double HalfY, double HalfZ, Point3 Center,int TextureType)
{
	m_AngleX = AngleX;
	m_AngleY = AngleY;
	m_AngleZ = AngleZ;
	m_HalfX = HalfX;
	m_HalfY = HalfY;
	m_HalfZ = HalfZ;
	m_Center = Center;

	m_TextureType = TextureType;

	// now figure out where the vertices are there should be eight of them
	// do the top ones first going clockwise
	m_Vertices[0].x = + HalfX;
	m_Vertices[0].y = + HalfY;
	m_Vertices[0].z = + HalfZ;

	m_Vertices[1].x = + HalfX;
	m_Vertices[1].y = + HalfY;
	m_Vertices[1].z = - HalfZ;

	m_Vertices[2].x = - HalfX;
	m_Vertices[2].y = + HalfY;
	m_Vertices[2].z = - HalfZ;

	m_Vertices[3].x = - HalfX;
	m_Vertices[3].y = + HalfY;
	m_Vertices[3].z = + HalfZ;

	// do the bottoms ones going clockwise
	m_Vertices[4].x = + HalfX;
	m_Vertices[4].y = - HalfY;
	m_Vertices[4].z = + HalfZ;

	m_Vertices[5].x = + HalfX;
	m_Vertices[5].y = - HalfY;
	m_Vertices[5].z = - HalfZ;

	m_Vertices[6].x = - HalfX;
	m_Vertices[6].y = - HalfY;
	m_Vertices[6].z = - HalfZ;

	m_Vertices[7].x = - HalfX;
	m_Vertices[7].y = - HalfY;
	m_Vertices[7].z = + HalfZ;

	m_BoundingSphere.enumStartAddVertices();
	m_BoundingAABB.enumStartAddVertices();

	// now do the rotations
	for (int ix = 0; ix < 8; ix++)
	{
		Point3 OldV = m_Vertices[ix];  // use this so the first and second equations
									   // have consistant values
		// do the x rotations first
		m_Vertices[ix].y = OldV.y*cos(m_AngleX*PI/180.0) - OldV.z*sin(m_AngleX*PI/180.0);
		m_Vertices[ix].z = OldV.y*sin(m_AngleX*PI/180.0) + OldV.z*cos(m_AngleX*PI/180.0);

		OldV = m_Vertices[ix];

		// do the y roations next
		m_Vertices[ix].x = OldV.x*cos(m_AngleY*PI/180.0) + OldV.z*sin(m_AngleY*PI/180.0);
		m_Vertices[ix].z = - OldV.x*sin(m_AngleY*PI/180.0) + OldV.z*cos(m_AngleY*PI/180.0);

		OldV = m_Vertices[ix];

		// do the z roations last
		m_Vertices[ix].x = OldV.x*cos(m_AngleZ*PI/180.0) - OldV.y*sin(m_AngleZ*PI/180.0);
		m_Vertices[ix].y = OldV.x*sin(m_AngleZ*PI/180.0) + OldV.y*cos(m_AngleZ*PI/180.0);

		// now do the translation
		m_Vertices[ix].x += Center.x; 
		m_Vertices[ix].y += Center.y;
		m_Vertices[ix].z += Center.z;

		// now form the bounding boxes
		m_BoundingSphere.enumAddVertices(m_Vertices[ix]);
		m_BoundingAABB.enumAddVertices(m_Vertices[ix]);
	}

	m_BoundingSphere.FormSphere();
	m_BoundingAABB.FormAABB();

	// now form the noralized vectors 
	Point3 v1,v2;
	// figure out the x plane vector
	v1 = m_Vertices[5] - m_Vertices[1];
	v2 = m_Vertices[1] - m_Vertices[0];
	Av = v2^v1; // cross product
	Av.Normalize();

	// figure out the y plane vector
	v1 = m_Vertices[0] - m_Vertices[1];
	v2 = m_Vertices[0] - m_Vertices[3];
	Aw = v2^v1;
	Aw.Normalize();

	// figure out the z plane vector
	v1 = m_Vertices[0] - m_Vertices[3];
	v2 = m_Vertices[0] - m_Vertices[4];
	Au = v2^v1;
	Au.Normalize();

//	GetLightMaps();

}

//////////////////////////////////////////////////////////////////////
// DirectXRender(TLNVERTEXSTRUCT *&pVertex, int &VertexCount)
//
// Render the box using direct x
//////////////////////////////////////////////////////////////////////
void Box::DirectXRender(TLNVERTEXSTRUCT *&pVertex, int &VertexCount, int *&pLightmap,int &LightMapCount)
{
	DWORD Color;
	if (m_objectSelected)
		Color = 0xffffffff;
	else
		Color = 0xffff0000;

	pLightmap[0] = lightMapIndex[0];
	pLightmap[1] = lightMapIndex[0];
	pLightmap[2] = lightMapIndex[1];
	pLightmap[3] = lightMapIndex[1];
	pLightmap[4] = lightMapIndex[2];
	pLightmap[5] = lightMapIndex[2];
	pLightmap[6] = lightMapIndex[3];
	pLightmap[7] = lightMapIndex[3];
	pLightmap[8] = lightMapIndex[4];
	pLightmap[9] = lightMapIndex[4];
	pLightmap[10] = lightMapIndex[5];
	pLightmap[11] = lightMapIndex[5];
	pLightmap += 12;
	LightMapCount += 12;



	// form top box
	DirectXFormSquare(pVertex,VertexCount
		,m_Vertices[3],m_Vertices[0],m_Vertices[1],m_Vertices[2],Color);

	// form bottom box
	DirectXFormSquare(pVertex,VertexCount
		,m_Vertices[4],m_Vertices[7],m_Vertices[6],m_Vertices[5],Color);
	
	DirectXFormSquare(pVertex,VertexCount
		,m_Vertices[1],m_Vertices[0],m_Vertices[4],m_Vertices[5],Color);

	DirectXFormSquare(pVertex,VertexCount
		,m_Vertices[2],m_Vertices[1],m_Vertices[5],m_Vertices[6],Color);

	DirectXFormSquare(pVertex,VertexCount
		,m_Vertices[3],m_Vertices[2],m_Vertices[6],m_Vertices[7],Color);

	DirectXFormSquare(pVertex,VertexCount
		,m_Vertices[0],m_Vertices[3],m_Vertices[7],m_Vertices[4],Color);

}

//////////////////////////////////////////////////////////////////////
// DirectXFormSquare(TLNVERTEXSTRUCT *&pVertex, int &VertexCount, Point3 v1, Point3 v2, Point3 v3, Point3 v4)
//
// Helper function forms a square out of 4 vertices
// Render the box using direct x
//
//
// by convention for texturing (x,y) = (u,v)
// v1 is top left (0,0)
// v2 is top right (1,0)
// v3 is bottom right (1,1)
// v4 is bottom left (0,1)
//////////////////////////////////////////////////////////////////////
void Box::DirectXFormSquare(TLNVERTEXSTRUCT *&pVertex, int &VertexCount, Point3 v1, Point3 v2, Point3 v3, Point3 v4, DWORD Color)
{

//	DWORD Color = 0xffffffff;
	Point3 V1,V2;

	V1.x = v2.x - v1.x;
	V1.y = v2.y - v1.y;
	V1.z = v2.z - v1.z;
	V2.x = v3.x - v1.x;
	V2.y = v3.y - v1.y;
	V2.z = v3.z - v1.z;
	// render the box here as 
	pVertex[0].x = v1.x;
	pVertex[0].y = v1.y;
	pVertex[0].z = v1.z;
	pVertex[0].color = Color;
	pVertex[0].color2 = Color;
	pVertex[0].tu = 0.0f;
	pVertex[0].tv = 0.0f;
	pVertex[0].tu2 = 0.0f;
	pVertex[0].tv2 = 0.0f;
	float N1 = V1.y * V2.z - V1.z * V2.y;
	float N2 = V1.z * V2.x - V1.x * V2.z;
	float N3 = V1.x * V2.y - V1.y * V2.x;

	pVertex[0].nx = N1/(abs(N1)+abs(N2)+abs(N3));
	pVertex[0].ny = N2/(N1+N2+N3); 
	pVertex[0].nz = N3/(N1+N2+N3);

	pVertex[1].x = v2.x;
	pVertex[1].y = v2.y;
	pVertex[1].z = v2.z;
	pVertex[1].color = Color;
	pVertex[1].color2 = Color;
	pVertex[1].tu = 1.0f;
	pVertex[1].tv = 0.0f;
	pVertex[1].tu2 = 1.0f;
	pVertex[1].tv2 = 0.0f;
	pVertex[1].nx = pVertex[0].nx;
	pVertex[1].ny = pVertex[0].ny;
	pVertex[1].nz = pVertex[0].nz;

	pVertex[2].x = v3.x;
	pVertex[2].y = v3.y;
	pVertex[2].z = v3.z;
	pVertex[2].color = Color;
	pVertex[2].color2 = Color;
	pVertex[2].tu = 1.0f;
	pVertex[2].tv = 1.0f;
	pVertex[2].tu2 = 1.0f;
	pVertex[2].tv2 = 1.0f;
	pVertex[2].nx = pVertex[0].nx;
	pVertex[2].ny = pVertex[0].ny;
	pVertex[2].nz = pVertex[0].nz;

	pVertex[3].x = v3.x;
	pVertex[3].y = v3.y;
	pVertex[3].z = v3.z;
	pVertex[3].color = Color;
	pVertex[3].color2 = Color;
	pVertex[3].tu = 1.0f;
	pVertex[3].tv = 1.0f;
	pVertex[3].tu2 = 1.0f;
	pVertex[3].tv2 = 1.0f;
	pVertex[3].nx = pVertex[0].nx;
	pVertex[3].ny = pVertex[0].ny;
	pVertex[3].nz = pVertex[0].nz;

	pVertex[4].x = v4.x;
	pVertex[4].y = v4.y;
	pVertex[4].z = v4.z;
	pVertex[4].color = Color;
	pVertex[4].color2 = Color;
	pVertex[4].tu = 0.0f;
	pVertex[4].tv = 1.0f;
	pVertex[4].tu2 = 0.0f;
	pVertex[4].tv2 = 1.0f;
	pVertex[4].nx = pVertex[0].nx;
	pVertex[4].ny = pVertex[0].ny;
	pVertex[4].nz = pVertex[0].nz;

	pVertex[5].x = v1.x;
	pVertex[5].y = v1.y;
	pVertex[5].z = v1.z;
	pVertex[5].color = Color;
	pVertex[5].color2 = Color;
	pVertex[5].tu = 0.0f;
	pVertex[5].tv = 0.0f;
	pVertex[5].tu2 = 0.0f;
	pVertex[5].tv2 = 0.0f;
	pVertex[5].nx = pVertex[0].nx;
	pVertex[5].ny = pVertex[0].ny;
	pVertex[5].nz = pVertex[0].nz;

	VertexCount += 2;
	pVertex+=6;
	

}

bool Box::RayOBBIntersect(Point3 Org, Point3 Dir, Point3 &Intersect)
{
	double tmin = -9999999999.9;
	double tmax = 9999999999.9;
	Point3 p = m_Center - Org;

		// z = u
	// w = y
	// v = x;

	Point3 Normals[3];
	double HalfV[3];
	Normals[0] = Au;
	Normals[1] = Av;
	Normals[2] = Aw;
	HalfV[0] = m_HalfZ;
	HalfV[1] = m_HalfX;
	HalfV[2] = m_HalfY;

	for (int ix = 0; ix < 3; ix++)
	{

		double e = Point3::Dot(Normals[ix],p);
		double f = Point3::Dot(Normals[ix],Dir);
		
		double af = f;

		ABS(af);

		if (af > SMALLE)
		{
			double t1 = (e + HalfV[ix])/f;
			double t2 = (e - HalfV[ix])/f;
			if (t1 > t2)
			{
				double t = t1;
				t1 = t2;
				t2 = t;
			}
			if (t1 > tmin) 
				tmin = t1;
			if (t2 < tmax)
				tmax = t2;
			if (tmin > tmax)
				return false;
			if (tmax < 0)
				return false;
		}
		else if ((( -e - HalfV[ix]) > 0) || ((-e + HalfV[ix]) < 0))
			return false;
	}

	if (tmin > 0)
		Intersect = Org + Dir*tmin;
	else
		Intersect = Org + Dir*tmax;

	return true;

}

bool Box::RayIntersectObject(Point3 &Org, Point3 &Dir)
{
	// check each triangle against the ray to see if it intersects

	if (IntersectTriangle(Org,Dir,m_Vertices[0],m_Vertices[1],m_Vertices[2]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[2],m_Vertices[3],m_Vertices[0]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[7],m_Vertices[6],m_Vertices[5]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[5],m_Vertices[4],m_Vertices[7]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[0],m_Vertices[4],m_Vertices[5]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[5],m_Vertices[1],m_Vertices[0]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[2],m_Vertices[3],m_Vertices[0]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[1],m_Vertices[5],m_Vertices[6]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[6],m_Vertices[2],m_Vertices[1]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[2],m_Vertices[6],m_Vertices[7]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[7],m_Vertices[3],m_Vertices[2]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[3],m_Vertices[7],m_Vertices[4]))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[4],m_Vertices[0],m_Vertices[3]))
		return true;


	return false;

}

// get the light maps
void Box::GetLightMaps()
{
		// now create the lightmaps
	lightMapIndex[0] = theLightMaps.CreateLightMap(
		m_Vertices[3],m_Vertices[0],m_Vertices[1],m_Vertices[2]);
	lightMapIndex[1] = theLightMaps.CreateLightMap(
		m_Vertices[7],m_Vertices[6],m_Vertices[5],m_Vertices[4]);
	lightMapIndex[2] = theLightMaps.CreateLightMap(
		m_Vertices[0],m_Vertices[4],m_Vertices[5],m_Vertices[1]);
	lightMapIndex[3] = theLightMaps.CreateLightMap(
		m_Vertices[1],m_Vertices[5],m_Vertices[6],m_Vertices[2]);
	lightMapIndex[4] = theLightMaps.CreateLightMap(
		m_Vertices[2],m_Vertices[6],m_Vertices[7],m_Vertices[3]);
	lightMapIndex[5] = theLightMaps.CreateLightMap(
		m_Vertices[3],m_Vertices[7],m_Vertices[4],m_Vertices[0]);

}

// checks all 12 triangles of box for collision this verison returns the intersection point
bool Box::RayIntersectObject(Point3 &Org, Point3 &Dir, Point3 &intersection)
{
	if (IntersectTriangle(Org,Dir,m_Vertices[0],m_Vertices[1],m_Vertices[2],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[2],m_Vertices[3],m_Vertices[0],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[7],m_Vertices[6],m_Vertices[5],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[5],m_Vertices[4],m_Vertices[7],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[0],m_Vertices[4],m_Vertices[5],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[5],m_Vertices[1],m_Vertices[0],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[2],m_Vertices[3],m_Vertices[0],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[1],m_Vertices[5],m_Vertices[6],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[6],m_Vertices[2],m_Vertices[1],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[2],m_Vertices[6],m_Vertices[7],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[7],m_Vertices[3],m_Vertices[2],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[3],m_Vertices[7],m_Vertices[4],intersection))
		return true;

	if (IntersectTriangle(Org,Dir,m_Vertices[4],m_Vertices[0],m_Vertices[3],intersection))
		return true;


	return false;
}
