// SpinEditBox.cpp : implementation file
//

#include "stdafx.h"
#include "MapEditor.h"
#include "SpinEditBox.h"
#include "EditFloat.h"
#include "EditInt.h"
#include "TerrainManager.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// SpinEditBox

SpinEditBox::SpinEditBox()
{
}

//==================================================================================
// SpinEditBox(char *Title, CRect Position, float Min, float Max, int Increments, BOOL isFloat, CWnd *Parent)
//
// Usage: This or Create should always be used in constructing a SpinEditBoxControl
//
// Notes: The width of Position should be at least 70 otherwise it won't look good
//
//==================================================================================
SpinEditBox::SpinEditBox(char *Title, CPoint Position, float Min, float Max, int Increments, UINT Flags, CWnd *Parent, int nID)
{
	Create(Title, Position,Min,Max,Increments,Flags,Parent,nID);

}

//==================================================================================
// Create(char *Title, CRect Position, float Min, float Max, int Increments, BOOL isFloat, CWnd *Parent)
//
// Usage: Create the control
//
// Notes: The size is determined by the defaults, however its position isn't
//		   the point given by Offset could either be the right most of left most point depending
//        if it is right or left jusitified
//
//==================================================================================
void SpinEditBox::Create(char *Title, CPoint Offset, float Min, float Max, int Increments, UINT Flags, CWnd *Parent, int nID)
{
		// copy over the values
	this->Min = Min;
	this->Max = Max;
	this->Increments = Increments;
	if (Flags & SE_FLOAT)
		this->isFloat = true;
	else
		this->isFloat = false;
	this->ParentPtr = Parent;
	this->ControlID = nID;

	this->Title = Title;

	int StdSpinHeight = 18;
	int StdSpinWidth = 80; // everything but the text
	int FontWidth = 5;
	int StdSize = 170; // for StdSize justification
	int TextSpacing = 4;

	MaxChar = 7; // up to 5 characters

	CurrentValueF = 0.f;
	CurrentValueI = 0;
	PrevUp = false;
	ManualChange = false;
	ManualChangeNI = false;

	// now register this window as a child
	char windowname[40]; 

    strcpy(windowname,AfxRegisterWndClass(CS_HREDRAW|CS_VREDRAW|CS_DBLCLKS, 
		        ::LoadCursor(NULL, IDC_ARROW), HBRUSH(COLOR_WINDOW+1), NULL));

    // Set the window's initial style
    DWORD m_dwWindowStyle = WS_VISIBLE | WS_CHILD;

	CRect CreationRect;
	CreationRect.top = Offset.y;
	CreationRect.bottom = Offset.y + StdSpinHeight;
	if (Flags & SE_RIGHTJUSTIFIED)
	{
		CreationRect.right = Offset.x ;
		CreationRect.left = Offset.x - StdSpinWidth - FontWidth*strlen(Title) - TextSpacing;
	}
	else // std spacing justified
	{
		CreationRect.left = Offset.x;
		CreationRect.right = Offset.x + StdSize;
	}

    // Create the render window 
    CreateEx(0  ,windowname,"GenericSpinEditWindow",m_dwWindowStyle,CreationRect,ParentPtr,1,NULL);
	WindowRect = CreationRect; // record for any future queries

	// now create the control
	// now create the spin and edit control
	int SpinnerSize = 16;
	int ButtonSize = 20;
	int EditBoxSize = 44;

	// Relative offsets now
	CreationRect.top = 0;
	CreationRect.bottom = StdSpinHeight;
	Offset.x -= WindowRect.left;

	if (Flags & SE_RIGHTJUSTIFIED)
	{
		CreationRect.right = Offset.x - ButtonSize;
		CreationRect.left = Offset.x - ButtonSize - SpinnerSize;
	}
	else // std spacing justified
	{
		CreationRect.right = Offset.x + StdSize - ButtonSize;
		CreationRect.left = Offset.x + StdSize - ButtonSize - SpinnerSize;
	}

	SpinButtonCtrl.Create(WS_CHILD | WS_VISIBLE,CreationRect,this,SPINCONTROLID);
	// now figure out the range of the spin control
	SpinButtonCtrl.SetRange(0,Increments);
	UDACCEL Accel[3];
	Accel[0].nSec = 0;
	Accel[0].nInc = 1;
	Accel[1].nSec = 2;
	Accel[1].nInc = 2;
	Accel[2].nSec = 5;
	Accel[2].nInc = 5;
	SpinButtonCtrl.SetAccel(3,Accel);
	if (isFloat)	
		SpinButtonCtrl.SetPos((int)(Increments*CurrentValueF/(Max-Min)));	
	else	
		SpinButtonCtrl.SetPos((int)(Increments*CurrentValueI/(Max-Min)));

	int Pos = SpinButtonCtrl.GetPos();

	// now create the edit control
	if (Flags & SE_RIGHTJUSTIFIED)
	{
		CreationRect.right = Offset.x - ButtonSize - SpinnerSize;
		CreationRect.left = Offset.x - ButtonSize - SpinnerSize - EditBoxSize;
	}
	else // std spacing justified
	{
		CreationRect.left = Offset.x + StdSize - ButtonSize - SpinnerSize - EditBoxSize;
		CreationRect.right = Offset.x + StdSize - ButtonSize - SpinnerSize;
	}
	Edit.Create(WS_CHILD | WS_VISIBLE | WS_BORDER | ES_READONLY | ES_LEFT,CreationRect,this,EDITCONTROLID);
	Edit.SetLimitText(MaxChar);
	Edit.SetFont(Fonts.GetFont(MSSANSSERIF));

	// create the manual edit button
	if (Flags & SE_RIGHTJUSTIFIED)
	{
		CreationRect.right = Offset.x;
		CreationRect.left = Offset.x - ButtonSize;
	}
	else // std spacing justified
	{
		CreationRect.left = Offset.x + StdSize - ButtonSize;
		CreationRect.right = Offset.x + StdSize;
	}

	ManualEdit.Create("M",BS_PUSHBUTTON | WS_VISIBLE | WS_CHILD, CreationRect,this,BUTTONCONTROLID);
	ManualEdit.SetFont(Fonts.GetFont(MSSANSSERIF));

	// updates the value (of the Cedit that is)
	DoNotSend = true;
	UpdateValue();

	theTerrainManager.RenderFlag++;

}

SpinEditBox::~SpinEditBox()
{
}


BEGIN_MESSAGE_MAP(SpinEditBox, CWnd)
	//{{AFX_MSG_MAP(SpinEditBox)
	ON_NOTIFY(UDN_DELTAPOS, SPINCONTROLID, OnSpinChanged)
	ON_BN_CLICKED(BUTTONCONTROLID,OnManualEdit)
	ON_WM_ERASEBKGND()
	ON_WM_PAINT()
		ON_WM_KILLFOCUS( )
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// SpinEditBox message handlers
// this function not used simpy because for some reason "." doesn't work in CEdit


//==================================================================================
// OnSpinChanged(NMHDR* pNMHDR, LRESULT* pResult) 
//
// Usage: Message handler for when the spin button is toggled, lots of extra code
//		  because the spin control is kind of flaky
//
//==================================================================================
void SpinEditBox::OnSpinChanged(NMHDR* pNMHDR, LRESULT* pResult) 
{
	NM_UPDOWN* pNMUpDown = (NM_UPDOWN*)pNMHDR;
	// TODO: Add your control notification handler code here
	int Position = pNMUpDown->iPos;

	if (ManualChange)
	{
		if (pNMUpDown->iDelta == 1)
		{
			PrevUp = true;
			Position++;
			if (ManualChangeNI) // Reset
				ManualChangeNI = false;
		}
		else
		{
			if (ManualChangeNI) // if between intervals don't change position until next time around
				ManualChangeNI = false;
			else
				Position--;
			PrevUp = false;
			
		}
		ManualChange = false;
	}
	else if (pNMUpDown->iDelta == 1 && PrevUp)	
		PrevUp = true;
	else if (pNMUpDown->iDelta == -1 && !PrevUp)
		PrevUp = false;
	else if(pNMUpDown->iDelta == 1 && !PrevUp)
	{
		if (Position == 0)
			Position = 1;
		else
			Position+=2;
		PrevUp = true;			
	}
	else if (pNMUpDown->iDelta == -1 && PrevUp)
	{
		int Upper,Lower;
		SpinButtonCtrl.GetRange(Lower,Upper);
		if (Position == Upper)		
			Position = Upper-1;
		else
			Position-=2;
		PrevUp = false;
		
	}

	if (Position > Increments)
		Position = Increments;
	SpinButtonCtrl.SetPos(Position);

	if (isFloat)	
		CurrentValueF = Position*(Max-Min)/Increments + Min;
	else
		CurrentValueI = (int)(Position*(Max-Min)/Increments + Min);

	UpdateValue();


	*pResult = 0;
}

//==================================================================================
// OnManualEdit()
//
// Usage: Message handler for when the manual edit button is clicked
//
// Note: Extra stuff (Manual Change BOOLs) due to flaky OnSpinChange
//==================================================================================
void SpinEditBox::OnManualEdit()
{
	if (isFloat)
	{		
		EditFloat tDialog;
		tDialog.m_Float = CurrentValueF;
		tDialog.SetRange(Min,Max);
		if (tDialog.DoModal() == IDOK)
		{	
			ManualChange = true;
			CurrentValueF = tDialog.m_Float;
			float Interval = (Max-Min)/Increments;
			if (CurrentValueF/Interval-(int)(CurrentValueF/Interval) != 0)
				ManualChangeNI = true;
			SpinButtonCtrl.SetPos((int)(Increments*CurrentValueF/(Max-Min)));
			UpdateValue();
		}
	}
	else
	{
		EditInt tDialog;
		tDialog.m_Int = CurrentValueI;
		tDialog.SetRange((int)Min,(int)Max);
		if (tDialog.DoModal() == IDOK)
		{
			ManualChange = true;
			CurrentValueI = tDialog.m_Int;
			float Interval = (Max-Min)/Increments;
			if ((float)CurrentValueI/(float)Interval-(int)(CurrentValueI/Interval) != 0)
				ManualChangeNI = true;
			SpinButtonCtrl.SetPos((int)(Increments*CurrentValueI/(Max-Min)));
			UpdateValue();
		}
	}
}

//==================================================================================
// UpdateValue()
//
// Usage: Called to update the value of the read only edit box, plus send a message back
//		  to the parent window saying a change has occured
//
//==================================================================================
void SpinEditBox::UpdateValue()
{
	char tempo[20];
	if (isFloat)
	{
		if (atof(tempo) > 100)
			sprintf(tempo,"%6.2f",CurrentValueF);
		else if (atof(tempo) > 10)
			sprintf(tempo,"%6.3f",CurrentValueF);
		else if (atof(tempo) < 1)
			sprintf(tempo,"%6.5f",CurrentValueF);
		else
			sprintf(tempo,"%6.4f",CurrentValueF);

		int Length = strlen(tempo);
		for (int ix = 0; ix < Length; ix++)
		{
			// replace a period with a comma
			if (tempo[ix] == 46)
			{
				tempo[ix] = 44;
			}
		}
		Edit.SetWindowText(tempo);
	}
	else
	{
		sprintf(tempo,"%d",CurrentValueI);
		Edit.SetWindowText(tempo);
	}

	if (!DoNotSend)
	{
		// send a message to the parent window
		::SendMessage(ParentPtr->m_hWnd,WM_ONUPDATESPINEDIT,(WPARAM)ControlID ,0);
	}
	DoNotSend = false;
}

//==================================================================================
// OnPaint() 
//
// Usage: Used only to draw the title of the control beside the controls
//
//==================================================================================
void SpinEditBox::OnPaint() 
{
	CPaintDC dc(this); // device context for painting

	int OldMode = dc.SetBkMode(TRANSPARENT);

	if (Title.GetLength() > 0)
	{
		CFont *OldFont = dc.SelectObject(Fonts.GetFont(MSSANSSERIF));
		dc.TextOut(0,2,Title);
		dc.SelectObject(OldFont);
	}
	
	dc.SetBkMode(OldMode);
}

//==================================================================================
// SetValue(float Value)
//
// Usage: Sets the value of the control, won't send it back to the parent window when
//        Updating
//
//==================================================================================
void SpinEditBox::SetValue(float Value)
{
	CurrentValueF = Value;	
	// bug if I don't do this
	if (Value != 0)
		ManualChange = true;
	else
		ManualChange = false;

	float Interval = (Max-Min)/Increments;
	if (CurrentValueF/Interval-(int)(CurrentValueF/Interval) != 0)
		ManualChangeNI = true; // in between an interval, caution if next movement is down
	SpinButtonCtrl.SetPos((int)(Increments*CurrentValueF/(Max-Min)));
	DoNotSend = true; // no point to send it back
	UpdateValue(); // just update the read only edit box

}

//==================================================================================
// SetValue(float int)
//
// Usage: Sets the value of the control, won't send it back to the parent window when
//        Updating
//
//==================================================================================
void SpinEditBox::SetValue(int Value)
{
	CurrentValueI = Value;

	// bug if I don't do this
	if (Value != 0)
		ManualChange = true;
	else
		ManualChange = false;			
	float Interval = (Max-Min)/Increments;
	if ((float)CurrentValueI/(float)Interval-(int)(CurrentValueI/Interval) != 0)
		ManualChangeNI = true; // in between an interval, caution if next movement is down
	SpinButtonCtrl.SetPos((int)(Increments*CurrentValueI/(Max-Min)));
	DoNotSend = true;
	UpdateValue();

}
//==================================================================================
// GetValue()
//
// Usage: Returns the current value
//
//==================================================================================
float SpinEditBox::GetValue()
{
	if (isFloat)
		return CurrentValueF;
	else
		return CurrentValueI;

}

//==================================================================================
// GetWindowRect()
//
// Usage: Returns the window
//
//==================================================================================
CRect SpinEditBox::GetWindowRect()
{
	return WindowRect;

}
