// TerrainManager.cpp: implementation of the TerrainManager class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "MapEditor.h"
#include "TerrainManager.h"
#include "Childview.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

TerrainManager theTerrainManager;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

TerrainManager::TerrainManager()
{
	HeadRenderingTilePtr = NULL;
	NumberofTiles = 0;
	Once = true;
	ChangeTerrainChanged = false;
	RenderingTerrainChanged = false;
	RenderFlag = 0;
}

TerrainManager::~TerrainManager()
{
	ResetTerrianManager();


}

//---------------------------------------------------------------------
// GenerateTerrian(int XSegments, int YSegments, int DepthLevel)
// 
// Usage: Called to generate a very basic starting terrain, all it does is takes
//        the inputs and create the tiles and add them to the octree
//
// Algorithm: The tiles are not stored in an array simply because tiles could be added or removed causing problems
//            However this function generates an array like tile mesh
//            1) Go through both Z and X however only store the Z in an temp array, so it could be references
//               when knitting together the tiles
//---------------------------------------------------------------------
void TerrainManager::GenerateTerrain(int XSegments, int ZSegments, float DepthLevel)
{
	TerrainValid = true;
	ResetTerrianManager();
	// first get the individual size (X and Z) of each tile
	OctTree::OctTreeInfo Info = theOctTree.GetOctTreeInfo();
	float XSize = (float)Info.Length/(float)XSegments;
	float ZSize = (float)Info.Width/(float)ZSegments;

	float Slope = 0; // create a level plane
//	float Slope = 2/(float)ZSegments; // go two unit up for the entire thing

	TileObject *ConstructArray[1000]; // hopefully in construction we won't have more than 1000 tiles in a row
	TileObject *PrevZTile = NULL;
	TileObject *CreateTile;

	for (int iz = 0; iz < ZSegments; iz++)
	{
		ConstructArray[iz] = NULL;
	}


	// from 0 -> whatever
	for (int ix = 0; ix < XSegments; ix++)
	{
		PrevZTile = NULL;
		// from point 0 -> whatever
		for (int iz = 0; iz < ZSegments; iz++)
		{
			// create the new tile object
			PLANE ConstructPlane;
			//		 Z  X
			// v1 is 0, 0
			ConstructPlane.v1.x = ix*XSize;
			ConstructPlane.v1.z = iz*ZSize;
			ConstructPlane.v1.y = DepthLevel+ix*Slope+iz*Slope;
			// v1 is 1, 0
			ConstructPlane.v2.x = ix*XSize;
			ConstructPlane.v2.z = (iz+1)*ZSize;
			ConstructPlane.v2.y = DepthLevel+ix*Slope + (iz+1)*Slope;
			// v1 is 1, 1
			ConstructPlane.v3.x = (ix+1)*XSize;
			ConstructPlane.v3.z = (iz+1)*ZSize;
			ConstructPlane.v3.y = DepthLevel+(ix+1)*Slope + (iz+1)*Slope;
			// v1 is 0, 1
			ConstructPlane.v4.x = (ix+1)*XSize;
			ConstructPlane.v4.z = iz*ZSize;
			ConstructPlane.v4.y = DepthLevel+(ix+1)*Slope +iz*Slope;

			CreateTile = new TileObject(ConstructPlane);

			if (PrevZTile != NULL)			
				CreateTile->CreateKnitPrevZ(PrevZTile);

			if (ConstructArray[iz] != NULL)			
				CreateTile->CreateKnitPrevX(ConstructArray[iz]);

			ConstructArray[iz] = CreateTile;
			PrevZTile = CreateTile;

			if (ix == 0 && iz == 0)
				HeadRenderingTilePtr = CreateTile;
		}
	}

	NumberofTiles = XSegments*ZSegments;
	InvalidateTerrainRender();

}		

//---------------------------------------------------------------------
// RenderTiles(LPDIRECT3DDEVICE8 m_pd3dDevice)
// 
// Usage: Renders the tiles, by using the HeadRenderingPtr and following its neighbours
//        in a zig zag pattern,
//
// Algorithm: 1) Use Triangle strips, start by going up the first Z neighbours
//			  2) Then go up an X neighbour, then continue backwards upwards the Z neighbours
//            3) Continue Zig Zag pattern
//---------------------------------------------------------------------
void TerrainManager::RenderTiles(LPDIRECT3DDEVICE8 m_pd3dDevice)
{
	if (HeadRenderingTilePtr == NULL)
		return;

	TileObject *CurrTilePtr = HeadRenderingTilePtr;
	TileObject *PrevTilePtr;


	if (Once)
	{
		Once = false;
	
		// The Vertex buffer is created on the first render call
		if( FAILED( m_pd3dDevice->CreateVertexBuffer( NumberofTiles*1.2*sizeof(VERTEX)*2+2,
												   D3DUSAGE_DYNAMIC ,
												   VERTEX_TYPE,
												   D3DPOOL_DEFAULT,
												   &g_pTileVB 
												   ) )) 
												   AfxMessageBox("Failed tile Vertex Buffer");
	}

	VERTEX* pVertex; 
	int VBIndex = 4;
	if( SUCCEEDED( g_pTileVB->Lock(0, 0, (BYTE**)&pVertex, 0 ) ) )
	{
		DWORD Color;

		if (CurrTilePtr->Selected)
			Color = 0xffff0000;
		else
			Color = 0xff349020;
		
		// create the first 4 points
		pVertex[0].position = CurrTilePtr->Plane.v1;
		pVertex[0].color = Color;
		pVertex[0].tu = 0.0f;
		pVertex[0].tv = 0.0f;

		pVertex[1].position = CurrTilePtr->Plane.v2;
		pVertex[1].color = Color;
		pVertex[1].tu = 1.0f;
		pVertex[1].tv = 1.0f;

		pVertex[2].position = CurrTilePtr->Plane.v4;
		pVertex[2].color = Color;
		pVertex[2].tu = 0.0f;
		pVertex[2].tv = 1.0f;

		pVertex[3].position = CurrTilePtr->Plane.v3;
		pVertex[3].color = Color;
		pVertex[3].tu = 0.0f;
		pVertex[3].tv = 0.0f;

		CurrTilePtr = CurrTilePtr->Neighbours[1];

		BOOL Foward = true;
		BOOL Transition = false;

		while (CurrTilePtr != NULL)
		{
			while (CurrTilePtr!= NULL)
			{
				if (CurrTilePtr->Selected)
					Color = 0xffff0000;
				else
					Color = 0xff349020;

				if (Foward)
				{
					if (Transition)
					{
						Transition = false;
						// write a dummy triangle to restablish the strip
						// O.K Now keep on travelling and adding to the strip
						pVertex[VBIndex].position = CurrTilePtr->Plane.v1;
						pVertex[VBIndex].color = Color;
						pVertex[VBIndex].tu = 0.0f;
						pVertex[VBIndex].tv = 0.0f;

						VBIndex++;

						// O.K Now keep on travelling and adding to the strip
						pVertex[VBIndex].position = CurrTilePtr->Plane.v2;
						pVertex[VBIndex].color = Color;
						pVertex[VBIndex].tu = 0.0f;
						pVertex[VBIndex].tv = 0.0f;
						VBIndex++;
					}


					// O.K Now keep on travelling and adding to the strip
					pVertex[VBIndex].position = CurrTilePtr->Plane.v4;
					pVertex[VBIndex].color = Color;
					pVertex[VBIndex].tu = 0.0f;
					pVertex[VBIndex].tv = 0.0f;

					VBIndex++;

					// O.K Now keep on travelling and adding to the strip
					pVertex[VBIndex].position = CurrTilePtr->Plane.v3;
					pVertex[VBIndex].color = Color;
					pVertex[VBIndex].tu = 0.0f;
					pVertex[VBIndex].tv = 0.0f;

					VBIndex++;

					PrevTilePtr = CurrTilePtr;
					CurrTilePtr = CurrTilePtr->Neighbours[1]; // go up the X
				}
				else
				{
					if (Transition)
					{
						Transition = false;
						// write a dummy triangle to restablish the strip
						// O.K Now keep on travelling and adding to the strip
						pVertex[VBIndex].position = CurrTilePtr->Plane.v3;
						pVertex[VBIndex].color = Color;
						pVertex[VBIndex].tu = 0.0f;
						pVertex[VBIndex].tv = 0.0f;

						VBIndex++;

						// O.K Now keep on travelling and adding to the strip
						pVertex[VBIndex].position = CurrTilePtr->Plane.v4;
						pVertex[VBIndex].color = Color;
						pVertex[VBIndex].tu = 0.0f;
						pVertex[VBIndex].tv = 0.0f;
						VBIndex++;
					}

					// O.K Now keep on travelling and adding to the strip
					pVertex[VBIndex].position = CurrTilePtr->Plane.v2;
					pVertex[VBIndex].color = Color;
					pVertex[VBIndex].tu = 0.0f;
					pVertex[VBIndex].tv = 0.0f;

					VBIndex++;

					// O.K Now keep on travelling and adding to the strip
					pVertex[VBIndex].position = CurrTilePtr->Plane.v1;
					pVertex[VBIndex].color = Color;
					pVertex[VBIndex].tu = 0.0f;
					pVertex[VBIndex].tv = 0.0f;

					VBIndex++;

					PrevTilePtr = CurrTilePtr;
					CurrTilePtr = CurrTilePtr->Neighbours[3]; // go down the X
				}

			}
			CurrTilePtr = PrevTilePtr->Neighbours[0]; // go up the Z
			if (Foward)
				Foward = false;
			else
			{			
				Foward = true;
			}
			Transition = true;
		}


		g_pTileVB->Unlock();
	}

				
	m_pd3dDevice->SetTexture(0,NULL);
			
//	m_pd3dDevice->SetRenderState(D3DRS_FILLMODE,D3DFILL_WIREFRAME);
	m_pd3dDevice->SetRenderState(D3DRS_FILLMODE,D3DFILL_SOLID);
//	m_pd3dDevice->SetRenderState(D3DRS_CULLMODE,D3DCULL_NONE); // no culling
	m_pd3dDevice->SetRenderState(D3DRS_CULLMODE,D3DCULL_CCW);
	 
   
	m_pd3dDevice->SetStreamSource( 0, g_pTileVB, sizeof(VERTEX) );
	m_pd3dDevice->SetVertexShader( VERTEX_TYPE ); 			 

	m_pd3dDevice->DrawPrimitive( D3DPT_TRIANGLESTRIP, 0, VBIndex-2);


}


//---------------------------------------------------------------------
// RenderTiles1(LPDIRECT3DDEVICE8 m_pd3dDevice)
// 
// Usage: Renders the tiles, by using the HeadRenderingPtr and following its neighbours
//        in a zig zag pattern but uses triangle list not strips
//
// Algorithm: 1) Use Triangle lists, start by going up the first Z neighbours
//			  2) Then go up an X neighbour, then continue backwards upwards the Z neighbours
//            3) Continue Zig Zag pattern
//---------------------------------------------------------------------
void TerrainManager::RenderTiles1(LPDIRECT3DDEVICE8 m_pd3dDevice)
{
	if (HeadRenderingTilePtr == NULL)
		return;

	TileObject *CurrTilePtr = HeadRenderingTilePtr;
	TileObject *PrevTilePtr;

	int ColorAlt = 0;
	int TotalColors = 6;
	DWORD SelectColor = 0xffff0000;
	DWORD Colors[6];
	Colors[0] = 0xff345213;
	Colors[1] = 0xff95f3e3;
	Colors[2] = 0xff69032f;
	Colors[3] = 0xff69f32f;
	Colors[4] = 0xff364b7e;
	Colors[5] = 0xfff9e32f;



	if (Once)
	{
		Once = false;
	
		// The Vertex buffer is created on the first render call
		if( FAILED( m_pd3dDevice->CreateVertexBuffer( NumberofTiles*sizeof(VERTEX)*6,
												   D3DUSAGE_DYNAMIC ,
												   VERTEX_TYPE,
												   D3DPOOL_DEFAULT,
												   &g_pTileVB 
												   ) )) 
												   AfxMessageBox("Failed tile Vertex Buffer");
	}

	if (RenderingTerrainChanged)
	{
		RenderingTerrainChanged = false;

		VERTEX* pVertex; 
		if( SUCCEEDED( g_pTileVB->Lock(0, 0, (BYTE**)&pVertex, 0 ) ) )
		{
			DWORD Color;

			if (CurrTilePtr->Selected)
				Color = SelectColor;
			else		
				Color = Colors[ColorAlt];

			ColorAlt++;
			if (ColorAlt >= TotalColors)		
				ColorAlt = 0;



			RenderFormPlane(pVertex,CurrTilePtr->Plane,Color);	

			CurrTilePtr = CurrTilePtr->Neighbours[1];

			BOOL Foward = true;

			while (CurrTilePtr != NULL)
			{
				while (CurrTilePtr!= NULL)
				{
						if (CurrTilePtr->Selected)
							Color = SelectColor;
						else		
							Color = Colors[ColorAlt];

						ColorAlt++;
						if (ColorAlt >= TotalColors)		
							ColorAlt = 0;

					RenderFormPlane(pVertex,CurrTilePtr->Plane,Color);			

					PrevTilePtr = CurrTilePtr;

					if (Foward)
						CurrTilePtr = CurrTilePtr->Neighbours[1]; // go up the X
					else
						CurrTilePtr = CurrTilePtr->Neighbours[3]; // go down the X

				}

				CurrTilePtr = PrevTilePtr->Neighbours[0]; // go up the Z	
				if (Foward)
					Foward = false;
				else
				{			
					Foward = true;
				}
			}


			g_pTileVB->Unlock();
		}
	}

				
	m_pd3dDevice->SetTexture(0,NULL);
			
//	m_pd3dDevice->SetRenderState(D3DRS_FILLMODE,D3DFILL_WIREFRAME);
	m_pd3dDevice->SetRenderState(D3DRS_FILLMODE,D3DFILL_SOLID);
//	m_pd3dDevice->SetRenderState(D3DRS_CULLMODE,D3DCULL_NONE); // no culling
	m_pd3dDevice->SetRenderState(D3DRS_CULLMODE,D3DCULL_CCW);
	 
   
	m_pd3dDevice->SetStreamSource( 0, g_pTileVB, sizeof(VERTEX) );
	m_pd3dDevice->SetVertexShader( VERTEX_TYPE ); 			 

	m_pd3dDevice->DrawPrimitive( D3DPT_TRIANGLELIST, 0, NumberofTiles*2);


}

//---------------------------------------------------------------------
// RenderFormPlane(VERTEX *pVertex, PLANE Plane, DWORD Color)
// 
// Usage: By RenderTiles1, forms a CCW Culled plane with two triangles
//		  Z  X
// v1 is  0  0 
// v2 is  1  0 
// v3 is  1  1  
// v4 is  0  1 
//---------------------------------------------------------------------
void TerrainManager::RenderFormPlane(VERTEX *&pVertex, PLANE Plane, DWORD Color)
{
	pVertex[0].position = Plane.v1;
	pVertex[0].color = Color;
	pVertex[0].tu = 0.0f;
	pVertex[0].tv = 0.0f;

	pVertex[1].position = Plane.v2;
	pVertex[1].color = Color;
	pVertex[1].tu = 0.0f;
	pVertex[1].tv = 0.0f;

	pVertex[2].position = Plane.v3;
	pVertex[2].color = Color;
	pVertex[2].tu = 0.0f;
	pVertex[2].tv = 0.0f;

	pVertex[3].position = Plane.v1;
	pVertex[3].color = Color;
	pVertex[3].tu = 0.0f;
	pVertex[3].tv = 0.0f;

	pVertex[4].position = Plane.v3;
	pVertex[4].color = Color;
	pVertex[4].tu = 0.0f;
	pVertex[4].tv = 0.0f;

	pVertex[5].position = Plane.v4;
	pVertex[5].color = Color;
	pVertex[5].tu = 0.0f;
	pVertex[5].tv = 0.0f;

	pVertex+=6;

}


//======================================================================
// ResetTerrianManager()
// 
// Usage: Destructure for the terrain manager
//
//======================================================================
void TerrainManager::ResetTerrianManager()
{
	HeadRenderingTilePtr = NULL;
	NumberofTiles = 0;
	PrevChangedFlag = 0;
	RenderFlag = 0;
	Once = true;
	
	 if (g_pTileVB != NULL)
         g_pTileVB->Release();
	 g_pTileVB = NULL;

}

//======================================================================
// ExecuteFunction(TileObject *CenterTile, const float ArgsPassed[], void (*pt2Func)(const float []))
// 
// Usage: Gut function which executes any function given by pt2Func, over the terrain mesh
//
// Algorithm: 1) Place the center tile in a stack
//		      2) As long as stack isn't empty remove, member, change height, and then add its neighbours
//			     into the stack, otherwise done
//            3) Goto 2)
//
//======================================================================
void TerrainManager::ExecuteFunction(TileObject *CenterTilePtr, const float ArgsPassed[], float (*pt2Func)(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[]))
{
	float NearZero = 0.001f; // close enough to zero to be zero
	// the stack used for execute function
	std::stack<TileObject*, std::vector<TileObject*> > Stack;

	D3DXVECTOR3 Center = CenterTilePtr->Plane.v1; // use v1 as the center (convention I guess)

	RenderFlag++; // increment the flag so tiles are only visited once

	int count = 0;

	Stack.push(CenterTilePtr);
	CenterTilePtr->RenderFlag = RenderFlag;
	BOOL Once = true;

	while (!Stack.empty())
	{
		TileObject *CurrTilePtr = Stack.top();
		Stack.pop();
		count++;

		if (CurrTilePtr->PrevChangedFlag != PrevChangedFlag)
		{
			// backup old plane information
		    CurrTilePtr->PrevChangedFlag = PrevChangedFlag;
			CurrTilePtr->PrevPlane = CurrTilePtr->Plane;
		}
					
		// now change the height of the current tile
		float Height[4];


		Height[0] = pt2Func(Center,CurrTilePtr->Plane.v1,ArgsPassed); // get the height
		Height[1] = pt2Func(Center,CurrTilePtr->Plane.v2,ArgsPassed); // get the height
		Height[2] = pt2Func(Center,CurrTilePtr->Plane.v3,ArgsPassed); // get the height
		Height[3] = pt2Func(Center,CurrTilePtr->Plane.v4,ArgsPassed); // get the height

		BOOL TileHigher;
		for (int ix = 0; ix < 4; ix++)
		{
			if (Height[ix] < NearZero)
				Height[ix] = -1;
		}
		// check the new tile height against the previous stored tile,
		// use the previous store tile if higher
	/*	if (CurrTilePtr->RenderFlag == (RenderFlag-1))
		{
			// if the tile was effected by the last action
			TileHigher = true;
		}*/
		if ((Height[0] <= CurrTilePtr->PrevPlane.v1.y)
			|| (Height[1] <= CurrTilePtr->PrevPlane.v2.y)
			|| (Height[2] <= CurrTilePtr->PrevPlane.v3.y)
			|| (Height[3] <= CurrTilePtr->PrevPlane.v4.y))
		{
			TileHigher = false;
		}
		else
		{ 
			TileHigher = true;
		}


		if (TileHigher)
		{
			// first add neighbours if valid to stack
			for (int ix = 0; ix < 4; ix++)
			{
				if (CurrTilePtr->Neighbours[ix] != NULL)
					if (CurrTilePtr->Neighbours[ix]->RenderFlag != RenderFlag)
					{
						Stack.push(CurrTilePtr->Neighbours[ix]);
						CurrTilePtr->Neighbours[ix]->RenderFlag = RenderFlag;
					}
			}
			// now adjust the height
			//CurrTilePtr->PrevChangedFlag = RenderFlag;
			CurrTilePtr->SetHeight1(Height); // finalize the new height
		}
		else
		{
			int Count = 0;
			// use backup tile heights if neccessary
			if (Height[0] < CurrTilePtr->PrevPlane.v1.y)
			{
				Height[0] =  CurrTilePtr->PrevPlane.v1.y;
				Count++;
			}
			if (Height[1] < CurrTilePtr->PrevPlane.v2.y)
			{
				Height[1] =  CurrTilePtr->PrevPlane.v2.y;
				Count++;
			}
			if (Height[2] < CurrTilePtr->PrevPlane.v3.y)
			{
				Height[2] =  CurrTilePtr->PrevPlane.v3.y;
				Count++;
			}
			if (Height[3] < CurrTilePtr->PrevPlane.v4.y)
			{
				Height[3] =  CurrTilePtr->PrevPlane.v4.y;
				Count++;
			}			

			// first add neighbours if valid to stack
			for (int ix = 0; ix < 4; ix++)
			{
				if (CurrTilePtr->Neighbours[ix] != NULL)
					// make sure it was not accessed before on this run
					if (CurrTilePtr->Neighbours[ix]->RenderFlag != RenderFlag)
					{
						// because we are in a decreasing function, make sure this
						// tile has already been effected when this function was increasing
						// otherwise we are waisting time
						if (CurrTilePtr->Neighbours[ix]->PrevChangedFlag == PrevChangedFlag
							|| Count != 4)
						{
							Stack.push(CurrTilePtr->Neighbours[ix]);
							CurrTilePtr->Neighbours[ix]->RenderFlag = RenderFlag;
						}
					}
			}

			CurrTilePtr->SetHeight1(Height); // finalize the new height			
		}
		
	}
	InvalidateTerrainRender();
	InvalidateTerrainChanged();

}

//======================================================================
// ExecuteFunctionB(TileObject *CenterTile, const float ArgsPassed[], void (*pt2Func)(const float []))
// 
// Usage: Gut function which executes any function given by pt2Func, over the terrain mesh
//        Same as normal one except add the tiles height to the return value of the function
//
// Algorithm: 1) Place the center tile in a stack
//		      2) As long as stack isn't empty remove, member, change height, and then add its neighbours
//			     into the stack, otherwise done
//            3) Goto 2)
//
//======================================================================
void TerrainManager::ExecuteFunctionB(TileObject *CenterTilePtr, const float ArgsPassed[], float (*pt2Func)(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[]))
{
	float NearZero = 0.001f; // close enough to zero to be zero
	float BaseHeight = CenterTilePtr->Plane.v1.y;
	// use previous if already touched
	if (CenterTilePtr->PrevChangedFlag == PrevChangedFlag)
		BaseHeight = CenterTilePtr->PrevPlane.v1.y;
	// the stack used for execute function
	std::stack<TileObject*, std::vector<TileObject*> > Stack;

	D3DXVECTOR3 Center = CenterTilePtr->Plane.v1; // use v1 as the center (convention I guess)

	RenderFlag++; // increment the flag so tiles are only visited once

	int count = 0;

	Stack.push(CenterTilePtr);
	CenterTilePtr->RenderFlag = RenderFlag;
	BOOL Once = true;

	while (!Stack.empty())
	{
		TileObject *CurrTilePtr = Stack.top();
		Stack.pop();
		count++;

		if (CurrTilePtr->PrevChangedFlag != PrevChangedFlag)
		{
			// backup old plane information
		    CurrTilePtr->PrevChangedFlag = PrevChangedFlag;
			CurrTilePtr->PrevPlane = CurrTilePtr->Plane;
		}
					
		// now change the height of the current tile
		float Height[4];


		Height[0] = pt2Func(Center,CurrTilePtr->Plane.v1,ArgsPassed) + BaseHeight; // get the height
		Height[1] = pt2Func(Center,CurrTilePtr->Plane.v2,ArgsPassed) + BaseHeight; // get the height
		Height[2] = pt2Func(Center,CurrTilePtr->Plane.v3,ArgsPassed) + BaseHeight; // get the height
		Height[3] = pt2Func(Center,CurrTilePtr->Plane.v4,ArgsPassed) + BaseHeight; // get the height

		BOOL TileHigher;
		for (int ix = 0; ix < 4; ix++)
		{
			if (Height[ix] < NearZero)
				Height[ix] = -1;
		}
		// check the new tile height against the previous stored tile,
		// use the previous store tile if higher
	/*	if (CurrTilePtr->RenderFlag == (RenderFlag-1))
		{
			// if the tile was effected by the last action
			TileHigher = true;
		}*/
		if ((Height[0] <= CurrTilePtr->PrevPlane.v1.y)
			|| (Height[1] <= CurrTilePtr->PrevPlane.v2.y)
			|| (Height[2] <= CurrTilePtr->PrevPlane.v3.y)
			|| (Height[3] <= CurrTilePtr->PrevPlane.v4.y))
		{
			TileHigher = false;
		}
		else
		{ 
			TileHigher = true;
		}


		if (TileHigher)
		{
			// first add neighbours if valid to stack
			for (int ix = 0; ix < 4; ix++)
			{
				if (CurrTilePtr->Neighbours[ix] != NULL)
					if (CurrTilePtr->Neighbours[ix]->RenderFlag != RenderFlag)
					{
						Stack.push(CurrTilePtr->Neighbours[ix]);
						CurrTilePtr->Neighbours[ix]->RenderFlag = RenderFlag;
					}
			}
			// now adjust the height
			//CurrTilePtr->PrevChangedFlag = RenderFlag;
			CurrTilePtr->SetHeight1(Height); // finalize the new height
		}
		else
		{
			int Count = 0;
			// use backup tile heights if neccessary
			if (Height[0] < CurrTilePtr->PrevPlane.v1.y)
			{
				Height[0] =  CurrTilePtr->PrevPlane.v1.y;
				Count++;
			}
			if (Height[1] < CurrTilePtr->PrevPlane.v2.y)
			{
				Height[1] =  CurrTilePtr->PrevPlane.v2.y;
				Count++;
			}
			if (Height[2] < CurrTilePtr->PrevPlane.v3.y)
			{
				Height[2] =  CurrTilePtr->PrevPlane.v3.y;
				Count++;
			}
			if (Height[3] < CurrTilePtr->PrevPlane.v4.y)
			{
				Height[3] =  CurrTilePtr->PrevPlane.v4.y;
				Count++;
			}			

			// first add neighbours if valid to stack
			for (int ix = 0; ix < 4; ix++)
			{
				if (CurrTilePtr->Neighbours[ix] != NULL)
					// make sure it was not accessed before on this run
					if (CurrTilePtr->Neighbours[ix]->RenderFlag != RenderFlag)
					{
						// because we are in a decreasing function, make sure this
						// tile has already been effected when this function was increasing
						// otherwise we are waisting time
						if (CurrTilePtr->Neighbours[ix]->PrevChangedFlag == PrevChangedFlag
							|| Count != 4)
						{
							Stack.push(CurrTilePtr->Neighbours[ix]);
							CurrTilePtr->Neighbours[ix]->RenderFlag = RenderFlag;
						}
					}
			}

			CurrTilePtr->SetHeight1(Height); // finalize the new height			
		}
		
	}
	InvalidateTerrainRender();
	InvalidateTerrainChanged();

}

//======================================================================
// ExecuteFunctionA(ATileObject *CenterTile, const float ArgsPassed[], void (*pt2Func)(const float []))
// 
// Usage: Gut function which executes any function given by pt2Func, over the terrain mesh
//        the "A" version is for additive functions not absolute ones
//
// Algorithm: 1) Place the center tile in a stack
//		      2) As long as stack isn't empty remove, member, change height, and then add its neighbours
//			     into the stack, otherwise done
//		      3) When executing function if return value is -9999 then that is an exit code, don't continue
//			     otherwise add the return value to the vertex value
//            3) Goto 2)
//
// Notes:
//	small error if period is 0.  Make sure this never happens
//======================================================================
void TerrainManager::ExecuteFunctionA(TileObject *CenterTilePtr, const float ArgsPassed[], float (*pt2Func)(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[]))
{
	std::stack<TileObject*, std::vector<TileObject*> > Stack;

	D3DXVECTOR3 Center = CenterTilePtr->Plane.v1; // use v1 as the center (convention I guess)

	RenderFlag++; // increment the flag so tiles are only visited once

	int count = 0;

	Stack.push(CenterTilePtr);
	CenterTilePtr->RenderFlag = RenderFlag;

	while (!Stack.empty())
	{
		TileObject *CurrTilePtr = Stack.top();
		Stack.pop();
		count++;

		if (CurrTilePtr->PrevChangedFlag != PrevChangedFlag)
		{
			// backup old plane information
		    CurrTilePtr->PrevChangedFlag = PrevChangedFlag;
			CurrTilePtr->PrevPlane = CurrTilePtr->Plane;
		}
					
		// now change the height of the current tile
		float DeltaHeight[4];

		DeltaHeight[0] = pt2Func(Center,CurrTilePtr->Plane.v1,ArgsPassed); // get the height
		DeltaHeight[1] = pt2Func(Center,CurrTilePtr->Plane.v2,ArgsPassed); // get the height
		DeltaHeight[2] = pt2Func(Center,CurrTilePtr->Plane.v3,ArgsPassed); // get the height
		DeltaHeight[3] = pt2Func(Center,CurrTilePtr->Plane.v4,ArgsPassed); // get the height

		float Height[4];
		int Count = 0;

		Height[0] = CurrTilePtr->PrevPlane.v1.y;
		Height[1] = CurrTilePtr->PrevPlane.v2.y;
		Height[2] = CurrTilePtr->PrevPlane.v3.y;
		Height[3] = CurrTilePtr->PrevPlane.v4.y;

		for (int ix = 0; ix < 4; ix++)
		{
			if (DeltaHeight[ix] != EXITCODE)
			{
				Count++;
				Height[ix] += DeltaHeight[ix];
			}
		}

		
		// first add neighbours if valid to stack
		for (ix = 0; ix < 4; ix++)
		{
			if (CurrTilePtr->Neighbours[ix] != NULL)
				// make sure it was not accessed before on this run
				if (CurrTilePtr->Neighbours[ix]->RenderFlag != RenderFlag)
				{
					// because we are in a decreasing function, make sure this
					// tile has already been effected when this function was increasing
					// otherwise we are waisting time
					if (CurrTilePtr->Neighbours[ix]->PrevChangedFlag == PrevChangedFlag
						|| Count != 0)
					{
						Stack.push(CurrTilePtr->Neighbours[ix]);
						CurrTilePtr->Neighbours[ix]->RenderFlag = RenderFlag;
					}
				}
		}
	
	
		CurrTilePtr->SetHeight1(Height); // finalize the new height		

	}
	InvalidateTerrainRender();
	InvalidateTerrainChanged();

}

//======================================================================
// OnSelectNewTile()
// 
// Usage: Increments counter which is used to make the current function (e.i the current tile
//        being editing, having an undo state, that it to reverse the change
//
//======================================================================
void TerrainManager::OnSelectNewTile()
{
	PrevChangedFlag++;

}

//======================================================================
// InvalidateTerrainChanged()
// 
// Usage: Sets a flag which tells the function RenderTiles to reload the vertex buffer
//        because a change occured 
//
//======================================================================
void TerrainManager::InvalidateTerrainRender()
{
	RenderingTerrainChanged = true;

}

//======================================================================
// ResetTerrainInOctree()
// 
// Usage: To go through every tile in the terrain manager, and fix its reference in the octree
//
// Why?  Because the large terrain tools don't do this so they run much quicker
// 
// Used By: When a tileeditview destructs, also when the user manually pressed a button on
//          the tile edit view
//======================================================================
void TerrainManager::ResetTerrainInOctree()
{
	// make sure this isn't the time this function is being called in the 
	// destructer loop
	if (TerrainValid)
	{
		if (ChangeTerrainChanged)
		{
			ChangeTerrainChanged = false;

			typedef std::list<StdObject*> OBJECTLIST;
			OBJECTLIST::iterator start = theOctTree.GetHeadOctNode()->ObjectList.begin();
			OBJECTLIST::iterator end = theOctTree.GetHeadOctNode()->ObjectList.end();

			OBJECTLIST AllTerrain;
			int Count = theOctTree.GetHeadOctNode()->ObjectList.size();
			while (start != end)
			{
				AllTerrain.push_back((*start));
				start++;
			}
			// now nuke the tile list
			theOctTree.ClearOctTree();

			// now put everything back in its new place
			start = AllTerrain.begin();
			end = AllTerrain.end();

			int Counter = 0;
			int Interval = 20; // 5 % a time
			int DesiredMod = Interval;
			theViewPtr->SetStatsPercent(0);

			while (start != end)
			{
				((TileObject *)(*start))->ResetinOctree2();
				Counter++;
				int Mod = Counter*100/Count;
				if (Mod == DesiredMod)
				{
					DesiredMod += Interval;
					if (DesiredMod < 100)
						theViewPtr->SetStatsPercent(DesiredMod);
				}
				start++;
			}
			theViewPtr->SetStatsPercent(100);
		}
	}
}

//======================================================================
// TerrainInvalid()
// 
// Usage: On the destructer loop of the octree could call ResetTerrainInOctree
//	      when parts of the octree are no longer valid.  
//	      This function causes ResetTerrainInOctree() not to work until the terrain 
//		  is reconstructed
//
//======================================================================
void TerrainManager::TerrainInvalid()
{
	TerrainValid = false;
}

//======================================================================
// InvalidateTerrainChanged()
// 
// Usage: Sets a flag which tells the function ResetTerrainInOctree to indeed reset it 
//        because of a change
//
//======================================================================
void TerrainManager::InvalidateTerrainChanged()
{
	ChangeTerrainChanged = true;

}
