// TileObject.cpp: implementation of the TileObject class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "MapEditor.h"
#include "TileObject.h"
#include "ChildView.h"
#include "TileEditView.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

TileObject::TileObject()
{

}

TileObject::TileObject(PLANE Plane)
{
	HasChange = false; // used in the terrain editor
	this->Plane = Plane;
	this->UAddPlanetoOctree(Plane); // add this to the octree
	AddGenerateBoundingSphere(); // generate the bounding sphere

	// null some values
	Neighbours[0] = NULL;
	Neighbours[1] = NULL;
	Neighbours[2] = NULL;
	Neighbours[3] = NULL;
}

TileObject::~TileObject()
{

}

//---------------------------------------------------------------------
// BOOL IntersectBoundingSphere(const Ray &Line)
// 
// Usage: First collision detection test, between a sphere and a line
//
// Algorithm, solves the equation of a sphere and a parameterized line
//
//---------------------------------------------------------------------
BOOL TileObject::IntersectBoundingSphere(const RAY &Ray, D3DXVECTOR3 &IntersectionPoint)
{
	double ax,bx,cx,
		  ay,by,cy,
		  az,bz,cz;

	D3DXVECTOR3 tReturn,ttReturn;

	LINE Line(Ray);	// we need a line not a ray	

	// We are solving for the parameterized t, to do this generate quadratic equation for it
	ax = (Line.Point1.x - Line.Point2.x)*(Line.Point1.x - Line.Point2.x);
	bx = 2*(Line.Point1.x - Line.Point2.x)*(Line.Point2.x - BoundingSphere.Center.x);
	cx = (Line.Point2.x - BoundingSphere.Center.x)*(Line.Point2.x - BoundingSphere.Center.x);

	ay = (Line.Point1.y - Line.Point2.y)*(Line.Point1.y - Line.Point2.y);
	by = 2*(Line.Point1.y - Line.Point2.y)*(Line.Point2.y - BoundingSphere.Center.y);
	cy = (Line.Point2.y - BoundingSphere.Center.y)*(Line.Point2.y - BoundingSphere.Center.y);

	az = (Line.Point1.z - Line.Point2.z)*(Line.Point1.z - Line.Point2.z);
	bz = 2*(Line.Point1.z - Line.Point2.z)*(Line.Point2.z - BoundingSphere.Center.z);
	cz = (Line.Point2.z - BoundingSphere.Center.z)*(Line.Point2.z - BoundingSphere.Center.z);

	double A,B,C;

	A = ax+ay+az;
	B = bx+by+bz;
	C = cx+cy+cz - BoundingSphere.Radius*BoundingSphere.Radius;

	// now solve quadratic formula
	double temp1 = B*B - 4*A*C;

	if (temp1 < 0)
		return false;

	// Now solve for the point
	float T1 = (-B + sqrt(temp1))/(2*A);
	if (T1 >= 0 && T1 <= 1)
	{
		tReturn.x = Line.Point1.x*T1 + Line.Point2.x*(1-T1);
		tReturn.y = Line.Point1.y*T1 + Line.Point2.y*(1-T1);
		tReturn.z = Line.Point1.z*T1 + Line.Point2.z*(1-T1);
		T1 = (-B - sqrt(temp1))/(2*A);
		if (T1 >= 0 && T1 <= 1)
		{
			// O.K now we have to figure out which point is closer
			ttReturn.x = Line.Point1.x*T1 + Line.Point2.x*(1-T1);
			ttReturn.y = Line.Point1.y*T1 + Line.Point2.y*(1-T1);
			ttReturn.z = Line.Point1.z*T1 + Line.Point2.z*(1-T1);
			float D1,D2;
			D1 = Math.GetDistance(Line.Point1,tReturn);
			D2 = Math.GetDistance(Line.Point1,ttReturn);

			if (D1 > D2)
				tReturn = ttReturn;
		}
		return true;
	}	

	T1 = (-B - sqrt(temp1))/(2*A);
	if (T1 >= 0 && T1 <= 1)
	{
		tReturn.x = Line.Point1.x*T1 + Line.Point2.x*(1-T1);
		tReturn.y = Line.Point1.y*T1 + Line.Point2.y*(1-T1);
		tReturn.z = Line.Point1.z*T1 + Line.Point2.z*(1-T1);
		return true;
	}

	return false;
}

//---------------------------------------------------------------------
// IntersectBoundingBox(const Ray &Line)
// 
// Usage: Calculates the intersection of a line with this objects bounding box level of collision detection
//
// Algorithm, uses ray - plane intersection math
//---------------------------------------------------------------------
BOOL TileObject::IntersectBoundingBox(const RAY &Line, D3DXVECTOR3 &IntersectionPoint)
{
	return Math.IntersectPlane(Plane,Line,IntersectionPoint);

}

//---------------------------------------------------------------------
// AddGenerateBoundingSphere()
// 
// Usage: When first creating the plane, to generate its bounding sphere info
//
// Algorithm, basically takes the midpoint of a plane, probably only works properly if plane
//            is symetric
//---------------------------------------------------------------------
void TileObject::AddGenerateBoundingSphere()
{
	float Extra = 0.5;
	D3DXVECTOR3 t1,t2; 
	t1.x = Plane.v1.x - (Plane.v1.x - Plane.v2.x)/2.0f;
	t1.y = Plane.v1.y - (Plane.v1.y - Plane.v2.y)/2.0f;
	t1.z = Plane.v1.z - (Plane.v1.z - Plane.v2.z)/2.0f;

	t2.x = Plane.v3.x - (Plane.v3.x - Plane.v4.x)/2.0f;
	t2.y = Plane.v3.y - (Plane.v3.y - Plane.v4.y)/2.0f;
	t2.z = Plane.v3.z - (Plane.v3.z - Plane.v4.z)/2.0f;

	BoundingSphere.Center.x = t1.x - (t1.x - t2.x)/2.0f;
	BoundingSphere.Center.y = t1.y - (t1.y - t2.y)/2.0f;
	BoundingSphere.Center.z = t1.z - (t1.z - t2.z)/2.0f;

	BoundingSphere.Radius = sqrt(
					(BoundingSphere.Center.x - Plane.v1.x)*(BoundingSphere.Center.x - Plane.v1.x) +
					(BoundingSphere.Center.y - Plane.v1.y)*(BoundingSphere.Center.y - Plane.v1.y) +
					(BoundingSphere.Center.z - Plane.v1.z)*(BoundingSphere.Center.z - Plane.v1.z)) + Extra;
}


//---------------------------------------------------------------------
// CreateKnitPrevX(TileObject *PrevX)
// 
// Usage: When the tile mesh is initially created, this knits the prev X tile to this one
//        e.g reference each other as neighbours
//---------------------------------------------------------------------
void TileObject::CreateKnitPrevX(TileObject *PrevX)
{
	// add each other as neighbours
	Neighbours[3] = PrevX;
	PrevX->Neighbours[1] = this;
}

//---------------------------------------------------------------------
// CreateKnitPrevZ(TileObject *PrevZ)
// 
// Usage: When the tile mesh is initially created, this knits the prev Z tile to this one
//        e.g reference each other as neighbours
//---------------------------------------------------------------------
void TileObject::CreateKnitPrevZ(TileObject *PrevZ)
{
	// add each other as neighbours
	Neighbours[2] = PrevZ;
	PrevZ->Neighbours[0] = this;

}

//---------------------------------------------------------------------
// OnSelect() 
// 
// Usage: The user has selected the object
//---------------------------------------------------------------------
void TileObject::OnSelect()
{
	// Create a tile edit dialog	
	EditView = new TileEditView;
	EditView->SetEditObject(this); // tell it to use this tile
	((TileEditView *)EditView)->Create(GetViewPtr()->GetEditViewWindow(),GetViewPtr());

	StdObject::OnSelect();
	// adjust the color scheme?
    theTerrainManager.InvalidateTerrainRender();


}

//---------------------------------------------------------------------
// OnDeselect()
// 
// Usage: The user has selected the object
//---------------------------------------------------------------------
void TileObject::OnDeselect()
{
	if (EditView != NULL)
	{
		delete EditView;
		EditView = NULL;
	}
	StdObject::OnDeselect();
	// adjust the color scheme?
	theTerrainManager.InvalidateTerrainRender();

}

//=====================================================================
// GetTileHeight(float Height[])
// 
// Usage: returns the tile height
//=====================================================================
void TileObject::GetTileHeight(float Height[])
{
	Height[0] = Plane.v1.y;
	Height[1] = Plane.v2.y;
	Height[2] = Plane.v3.y;
	Height[3] = Plane.v4.y;
}

//=====================================================================
// SetHeight1(float Height[])
// 
// Usage: Sets the height of a tile
//
// Algorithm: This version only change the currrent tile an not its neighbours
//
//=====================================================================
void TileObject::SetHeight1(float Height[])
{
// double check the height
	OctTree::OctTreeInfo Info = theOctTree.GetOctTreeInfo();

	float Min = 0;
	float Max = Info.Height;

	for (int ix = 0; ix < 4; ix++)
	{
		if (Height[ix] < Min)
			Height[ix] = Min;
		else if (Height[ix] > Max)
			Height[ix] = Max;
	}

	TileObject *tTileObject;

	// change height for 5			
	tTileObject = this;
	tTileObject->Plane.v1.y = Height[0];
	tTileObject->Plane.v2.y = Height[1];
	tTileObject->Plane.v3.y = Height[2];
	tTileObject->Plane.v4.y = Height[3];
//	tTileObject->ResetInOctree(); this will be performed ah la garbage collection time
	tTileObject->HasChange = true; // do this instead of reseting it in the octree
}

//=====================================================================
// SetHeight(float Height[])
// 
// Usage: Sets the height of a tile
//
// Algorithm: There are 8 other tiles effected by the height change so
//            Find them, change their height, and then readd them to the octree
//
//=====================================================================
void TileObject::SetHeight(float Height[])
{
	// double check the height
	OctTree::OctTreeInfo Info = theOctTree.GetOctTreeInfo();

	float Min = 0;
	float Max = Info.Height;

	for (int ix = 0; ix < 4; ix++)
	{
		if (Height[ix] < Min)
			Height[ix] = Min;
		else if (Height[ix] > Max)
			Height[ix] = Max;
	}

	// use telephone style in terms of reference from 1-9 (1 being bottom left, 9 being top right)
	// where X axis is left-right Z axis is down-top
	
	TileObject *tTileObject;

	// change height for 5			
	tTileObject = this;
	tTileObject->Plane.v1.y = Height[0];
	tTileObject->Plane.v2.y = Height[1];
	tTileObject->Plane.v3.y = Height[2];
	tTileObject->Plane.v4.y = Height[3];
	tTileObject->ResetInOctree();
                               
	// Changing height for 1
	if (Neighbours[2] != NULL)
	{
		if (Neighbours[2]->Neighbours[3] != NULL)
		{
			tTileObject = Neighbours[2]->Neighbours[3];

			tTileObject->Plane.v3.y = Height[0];
			tTileObject->ResetInOctree();
		}
	}	

	// Change height for 2
	if (Neighbours[2] != NULL)
	{
			tTileObject = Neighbours[2];
			tTileObject->Plane.v2.y = Height[0];
			tTileObject->Plane.v3.y = Height[3];
			tTileObject->ResetInOctree();
	}		
	
	// change height for 3
	if (Neighbours[2] != NULL)
	{
		if (Neighbours[2]->Neighbours[1] != NULL)
		{
			tTileObject = Neighbours[2]->Neighbours[1];
			tTileObject->Plane.v2.y = Height[3];
			tTileObject->ResetInOctree();
		}
	}

	// change height for 4
	if (Neighbours[3] != NULL)
	{		
		tTileObject = Neighbours[3];
		tTileObject->Plane.v3.y = Height[1];
		tTileObject->Plane.v4.y = Height[0];
		tTileObject->ResetInOctree();
	}	

	// change height for 6
	if (Neighbours[1] != NULL)
	{		
		tTileObject = Neighbours[1];
		tTileObject->Plane.v1.y = Height[3];
		tTileObject->Plane.v2.y = Height[2];
		tTileObject->ResetInOctree();
	}

	// change height for 7
	if (Neighbours[0] != NULL)
	{	
		if (Neighbours[0]->Neighbours[3] != NULL)
		{
			tTileObject = Neighbours[0]->Neighbours[3];
			tTileObject->Plane.v4.y = Height[1];			
			tTileObject->ResetInOctree();
		}
	}

	// change height for 8
	if (Neighbours[0] != NULL)
	{		
		tTileObject = Neighbours[0];
		tTileObject->Plane.v1.y = Height[1];
		tTileObject->Plane.v4.y = Height[2];
		tTileObject->ResetInOctree();
	}

	// change height for 9
	if (Neighbours[0] != NULL)
	{		
		if (Neighbours[0]->Neighbours[1] != NULL)
		{
			tTileObject = Neighbours[0]->Neighbours[1];
			tTileObject->Plane.v1.y = Height[2];
			tTileObject->ResetInOctree();
		}
	}
}

//=====================================================================
// ResetInOctree()
// 
// Usage: Called to reset (delete then add) a tile from the octree
//
//=====================================================================
void TileObject::ResetInOctree()
{
	URemoveObjectfromOctree();
	UAddPlanetoOctree(Plane); // add this to the octree
	AddGenerateBoundingSphere(); // generate the bounding sphere
}

//=====================================================================
// SetHeightD(float Height)
// 
// Usage: Called to increase the height of a tile by using a delta value
//
//=====================================================================
void TileObject::SetHeightD(float DeltaHeight)
{
	float Height[4];
	Height[0] = Plane.v1.y + DeltaHeight;
	Height[1] = Plane.v2.y + DeltaHeight;
	Height[2] = Plane.v3.y + DeltaHeight;
	Height[3] = Plane.v4.y + DeltaHeight;
	SetHeight(Height);
}

//=====================================================================
// ResetinOctree2()
// 
// Usage: Simular to ResetinOctree except the deletion has already occured so just delete
//        internal list, and add normally
//
//=====================================================================
void TileObject::ResetinOctree2()
{
	OctList.clear();
	UAddPlanetoOctree(Plane); // add this to the octree
	AddGenerateBoundingSphere(); // generate the bounding sphere

}
