// UMath.cpp: implementation of the UMath class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "MapEditor.h"
#include "UMath.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

UMath Math; // global reference

UMath::UMath()
{

}

UMath::~UMath()
{

}

//---------------------------------------------------------------------
// Log2(int intput)
// 
// Calculates the Log2 of a number;
// Returns int (Log2)
//---------------------------------------------------------------------
int UMath::Log2(int input)
{
	int count = 0;
	while (input >= 2)
	{
		input = input/2;
		count++;
	}
	return count;
}

//---------------------------------------------------------------------
// Pwr2(input power)
// 
// Calculates the Pwr2 of a number;
// Returns int (Pwr2)
//---------------------------------------------------------------------
int UMath::Pwr2(int power)
{
	int output = 1;
	for (int ix = 0; ix < power; ix++)
	{
		output *= 2;
	}
	return output;

}

/************************************************************************
* Function float Mod360(float Angle)
* 
* PURPOSE
* To module an angle by 360
*************************************************************************/
float UMath::Mod360(float Angle)
{    
    float tReturn = Angle;

    while (tReturn > 360)
    {
        tReturn = tReturn - 360;       
    }
    while (tReturn < 0)
    {
        tReturn = 360 + tReturn;
    }

    return tReturn;
}

/************************************************************************
* Function BOOL InRect(CPoint Point, CRect Rect)
* 
* returns if a point is in a rectangle
*************************************************************************/
BOOL UMath::InRect(CPoint Point, CRect Rect)
{
	    if ((Point.x >= Rect.left) && (Point.x <= Rect.right) && (Point.y >= Rect.top) && (Point.y <= Rect.bottom))
    {
        return true;
    }

    return false;

}

/************************************************************************
* Function Normlize(float in1, float in2, float sum)
* 
* Noralizes two numbers to a sum
*************************************************************************/
void UMath::Normalize(float &in1, float &in2, float sum)
{
	float NSum = abs(in1) + abs(in2);

	in1 = in1/NSum*sum;
	in2 = in2/NSum*sum;


}

/************************************************************************
* Function abs(float in)
* 
* returns abs of a float
*************************************************************************/
inline float UMath::abs(float in)
{
	if (in < 0)
		return -in;
	else 
		return in;
}

//---------------------------------------------------------------------
// GeneratePlane(Plane &One)
// 
// Usage: Generates the plane equation ax+by+cz+d = 0 for 4 verteces
//
// Note (Obsolete?);
//---------------------------------------------------------------------
void UMath::GeneratePlane(PLANE &One)
{
	// first create the plane equation given the verteces
	D3DXVECTOR3 v1,v2,cross_v1xv2;

	v1 = One.v2 - One.v1;
	v2 = One.v3 - One.v1;

	// figure out normal vector
	D3DXVec3Cross(&cross_v1xv2,&v1,&v2);

	One.pA = cross_v1xv2.x;
	One.pB = cross_v1xv2.y;
	One.pC = cross_v1xv2.z;

	One.pD = -D3DXVec3Dot(&cross_v1xv2, &One.v1);
}

//---------------------------------------------------------------------
// BOOL FindIntersection(D3DXVECTOR3 &Out, PLANE &Plane, D3DXVECTOR3 &v1, D3DXVECTOR3 &v2)
// 
// Usage: Finds the intersection between a line between two vertices (v1 and v2) with a plane
// given by Plane,
//
// Return true if the line intersection with the point being Out
//
// Note (Obsolete?);
//---------------------------------------------------------------------
BOOL UMath::FindIntersection(D3DXVECTOR3 &Out, PLANE &Plane, D3DXVECTOR3 &v1, D3DXVECTOR3 &v2)
{
	float t1,t2;

	t1 = Plane.pA*v1.x + Plane.pB*v1.y + Plane.pC*v1.z;
	t2 = Plane.pA*v2.x + Plane.pB*v2.y + Plane.pC*v2.z;

	// line is on the plane of course they intersect
	ASSERT(t1 != t2);
	if (t1 == t2)
	{
		if (Plane.pD == 0)
			return false;
		else
			return true;
	}
	

	float final_t = -(t2 + Plane.pD)/(t1-t2);

	if (final_t > 1 || final_t < 0)
		return false;

	Out.x = v1.x*final_t + v2.x*(1-final_t);
	Out.y = v1.y*final_t + v2.y*(1-final_t);
	Out.z = v1.z*final_t + v2.z*(1-final_t);

	return true;

}

//---------------------------------------------------------------------
// BOOL IntersectTriangle( const Triangle &triangle, const RAY& ray)
// 
// Usage: Finds the intersection between a line between two vertices (v1 and v2) with a 3D triangle
//
// Return true if the line intersection with the triangle
//
// Note, 
// 1)this is the basis of all non spherical collision detection, it is borrowed from the directX
//   SDK with some modifications from Tom.
// 2) This only checks one face of the triangle, as a result this function has to be called twice
//---------------------------------------------------------------------
BOOL UMath::IntersectTriangle( const Triangle &triangle, const RAY& Ray)
{
    // Find vectors for two edges sharing vert0
    D3DXVECTOR3 edge1 = triangle.v1 - triangle.v0;
    D3DXVECTOR3 edge2 = triangle.v2 - triangle.v0;

    // Begin calculating determinant

    D3DXVECTOR3 pvec;
    D3DXVec3Cross( &pvec, &Ray.direction, &edge2 );

    // If determinant is near zero, ray lies in plane of triangle
    FLOAT det = D3DXVec3Dot( &edge1, &pvec );
    if( det < 0.0001f ) return FALSE;

    // Calculate distance from vert0 to ray origin
    D3DXVECTOR3 tvec = Ray.origin - triangle.v0;

    // Calculate U parameter and test bounds
    float u = D3DXVec3Dot( &tvec, &pvec );
    if( u < 0.0f || u > det ) return FALSE;

    // Prepare to test V parameter
    D3DXVECTOR3 qvec;
    D3DXVec3Cross( &qvec, &tvec, &edge1 );

    // Calculate V parameter and test bounds
    float v = D3DXVec3Dot( &Ray.direction, &qvec );
    if( v < 0.0f || u + v > det ) return FALSE;

	return TRUE;
}

//---------------------------------------------------------------------
// BOOL IntersectTriangle( const Triangle &triangle, const RAY& ray, D3DXVECTOR3 &intersectPnt )
// 
// Usage: Finds the intersection between a line between two vertices (v1 and v2) with a 3D triangle
//
// Return true if the line intersection with the triangle
//
// Note, 
// 1)this is the basis of all non spherical collision detection, it is borrowed from the directX
//   SDK with some modifications from Tom.
// 2) This only checks one face of the triangle, as a result this function has to be called twice
//---------------------------------------------------------------------
BOOL UMath::IntersectTriangle( const Triangle &triangle, const RAY& Ray, D3DXVECTOR3 &intersectPnt)
{
    // Find vectors for two edges sharing vert0
    D3DXVECTOR3 edge1 = triangle.v1 - triangle.v0;
    D3DXVECTOR3 edge2 = triangle.v2 - triangle.v0;

    // Begin calculating determinant

    D3DXVECTOR3 pvec;
    D3DXVec3Cross( &pvec, &Ray.direction, &edge2 );

    // If determinant is near zero, ray lies in plane of triangle
    FLOAT det = D3DXVec3Dot( &edge1, &pvec );
    if( det < 0.0001f ) return FALSE;

    // Calculate distance from vert0 to ray origin
    D3DXVECTOR3 tvec = Ray.origin - triangle.v0;

    // Calculate U parameter and test bounds
    float u = D3DXVec3Dot( &tvec, &pvec );
    if( u < 0.0f || u > det ) return FALSE;

    // Prepare to test V parameter
    D3DXVECTOR3 qvec;
    D3DXVec3Cross( &qvec, &tvec, &edge1 );

    // Calculate V parameter and test bounds
    float v = D3DXVec3Dot( &Ray.direction, &qvec );
    if( v < 0.0f || u + v > det ) return FALSE;

	// The ray intersects the triangle, find the intersection point
	D3DXPLANE plane;
	D3DXPlaneFromPoints(&plane,&triangle.v0,&triangle.v1,&triangle.v2);

	D3DXVECTOR3 secondPoint = Ray.origin + Ray.direction;
	D3DXPlaneIntersectLine(&intersectPnt,&plane,&Ray.origin,&secondPoint);

    return TRUE;
}

//---------------------------------------------------------------------
// BOOL IntersectPlane(const PLANE &Plane1, const PLANE &Plane2, D3DXVECTOR3 &intersectPnt)
// 
// Usage: Finds the intersection between a two planes
//
// Return true if the plane intersects with the plane with the intersectionPnt being returned too.
//
// Algorithm
// 1) A plane is two triangles, split both planes into two triangles each.
// 2) to find the intersection of any two triangles (use helper function)
// 3) If intersection call another function to see if collision point is in the bounds of the line
// 4) Do this for each triangle (4 times)
//
// Note:
// By using IntersectionTraingle, this is perhaps doing 4x as many computations but the function was avialable.
//---------------------------------------------------------------------
BOOL UMath::IntersectPlane(PLANE &Plane1, PLANE &Plane2)
{
	Triangle Plane1a,Plane1b,Plane2a,Plane2b;	
	LINE DummyCollisionLine;
	D3DXVECTOR3 DummyVector;

	// form each of the four triangles out of the two planes
	Plane1a = Triangle(Plane1.v1,Plane1.v2,Plane1.v3);
	Plane1b = Triangle(Plane1.v1,Plane1.v4,Plane1.v3);
	Plane2a = Triangle(Plane2.v1,Plane2.v2,Plane2.v3);
	Plane2b = Triangle(Plane2.v1,Plane2.v4,Plane2.v3);

	GeneratePlane(Plane1);
	GeneratePlane(Plane2);



	if (IntersectTriangle(Plane1a,Plane2a,DummyCollisionLine,DummyVector))
		return true;

	if (IntersectTriangle(Plane1a,Plane2b,DummyCollisionLine,DummyVector)) 
		return true;

	if (IntersectTriangle(Plane1b,Plane2a,DummyCollisionLine,DummyVector)) 
		return true;

	if (IntersectTriangle(Plane1b,Plane2b,DummyCollisionLine,DummyVector)) 
		return true;
	
	return false;
}


//---------------------------------------------------------------------
// BOOL IntersectTriangle(const Triangle &T1, const Triangle &T2, RAY &CollisionRay, D3DXVECTOR3 &IntersectPoint)
// 
// Usage: Finds if two triangles intersect each other, helper function for IntersectPlane
//
// Return true if the triangles intersect each other.
//
// Algorithm
// 1) Check each line of both triangles (6 in total * 2 for both faces) to check if they intersect the other triangle
// 2) If one does then the two triangle intersect, return true, else return false.
//---------------------------------------------------------------------
inline BOOL UMath::IntersectTriangle(const Triangle &T1, const Triangle &T2, LINE &Line, D3DXVECTOR3 &IntersectPoint)
{
	
	// One Face
	Line.Point1 = T1.v0;
	Line.Point2 = T1.v1;

	if (IntersectTriangle(T2,Line,IntersectPoint))
		return true;

	Line.Point1 = T1.v1;
	Line.Point2 = T1.v2;

	if (IntersectTriangle(T2,Line,IntersectPoint))
		return true;

	Line.Point1 = T1.v2;
	Line.Point2 = T1.v0;

	if (IntersectTriangle(T2,Line,IntersectPoint))
		return true;

	// Other Face
	Triangle T3;
	T3.v0 = T2.v2;
	T3.v1 = T2.v1;
	T3.v2 = T2.v0;
	
	Line.Point1 = T1.v0;
	Line.Point2 = T1.v1;

	if (IntersectTriangle(T3,Line,IntersectPoint))
		return true;

	Line.Point1 = T1.v1;
	Line.Point2 = T1.v2;

	if (IntersectTriangle(T3,Line,IntersectPoint))
		return true;

	Line.Point1 = T1.v2;
	Line.Point2 = T1.v0;

	if (IntersectTriangle(T3,Line,IntersectPoint))
		return true;

	// One face

	Line.Point1 = T2.v0;
	Line.Point2 = T2.v1;

	if (IntersectTriangle(T1,Line,IntersectPoint))
		return true;

	Line.Point1 = T2.v1;
	Line.Point2 = T2.v2;

	if (IntersectTriangle(T1,Line,IntersectPoint))
		return true;

	Line.Point1 = T2.v2;
	Line.Point2 = T2.v0;

	if (IntersectTriangle(T1,Line,IntersectPoint))
		return true;

	// Other face
	T3.v0 = T1.v2;
	T3.v1 = T1.v1;
	T3.v2 = T1.v0;

	Line.Point1 = T2.v0;
	Line.Point2 = T2.v1;

	if (IntersectTriangle(T3,Line,IntersectPoint))
		return true;

	Line.Point1 = T2.v1;
	Line.Point2= T2.v2;

	if (IntersectTriangle(T3,Line,IntersectPoint))
		return true;

	Line.Point1 = T2.v2;
	Line.Point2 = T2.v0;

	if (IntersectTriangle(T1,Line,IntersectPoint))
		return true;

	return false;
}

//---------------------------------------------------------------------
// BOOL IntersectPlane(const PLANE &Plane, const LINE &Line, D3DXVECTOR3 &IntersectPoint)
// 
// Usage: Finds if a line and a plane intersect each other
//
// Return true if the triangles intersect each other.
//
// Algorithm
// 1) Do triangle check, X4
//---------------------------------------------------------------------
BOOL UMath::IntersectPlane(const PLANE &Plane, const LINE &Line, D3DXVECTOR3 &IntersectPoint)
{
	Triangle Plane1a,Plane1b;

	// form each of the four triangles out of the two planes
	Plane1a = Triangle(Plane.v1,Plane.v2,Plane.v3);
	Plane1b = Triangle(Plane.v1,Plane.v4,Plane.v3);

	if (IntersectTriangle(Plane1a,Line,IntersectPoint))
		return true;

	if (IntersectTriangle(Plane1b,Line,IntersectPoint))
		return true;
	
	// Other Face
	Triangle T3;
	T3.v0 = Plane1a.v2;
	T3.v1 = Plane1a.v1;
	T3.v2 = Plane1a.v0;

	if (IntersectTriangle(T3,Line,IntersectPoint))
		return true;

	// Other Face
	T3.v0 = Plane1b.v2;
	T3.v1 = Plane1b.v1;
	T3.v2 = Plane1b.v0;


	if (IntersectTriangle(T3,Line,IntersectPoint))
		return true;

	return false;
		

}

//---------------------------------------------------------------------
// BOOL IntersectPlane(const PLANE &Plane, const LINE &Line, D3DXVECTOR3 &IntersectPoint)
// 
// Usage: Finds if a line and a plane intersect each other
//
// Return true if the triangles intersect each other.
//
// Algorithm
// 1) Do triangle check, X4
//---------------------------------------------------------------------
BOOL UMath::IntersectPlane(const PLANE &Plane, const RAY &Ray, D3DXVECTOR3 &IntersectPoint)
{
	Triangle Plane1a,Plane1b;

	// form each of the four triangles out of the two planes
	Plane1a = Triangle(Plane.v1,Plane.v2,Plane.v3);
	Plane1b = Triangle(Plane.v1,Plane.v4,Plane.v3);

	if (IntersectTriangle(Plane1a,Ray,IntersectPoint))
		return true;

	if (IntersectTriangle(Plane1b,Ray,IntersectPoint))
		return true;

	// Other Face
	Triangle T3;
	T3.v0 = Plane1a.v2;
	T3.v1 = Plane1a.v1;
	T3.v2 = Plane1a.v0;

	// don't check the other face for now, only the CCW one

	if (IntersectTriangle(T3,Ray,IntersectPoint))
		return true;

	// Other Face
	T3.v0 = Plane1b.v2;
	T3.v1 = Plane1b.v1;
	T3.v2 = Plane1b.v0;


	if (IntersectTriangle(T3,Ray,IntersectPoint))
		return true;

	return false;
		

}
//---------------------------------------------------------------------
// float GetDistance(const D3DXVECTOR3 &PointOne, const D3DXVECTOR3 &PointTwo)
// 
// Usage: Returns the distance of two 3D points
//---------------------------------------------------------------------
float UMath::GetDistance(const D3DXVECTOR3 &PointOne, const D3DXVECTOR3 &PointTwo)
{
	return (float)sqrt((double)(PointOne.x-PointTwo.x)*(double)(PointOne.x-PointTwo.x)
		+(double)(PointOne.y-PointTwo.y)*(double)(PointOne.y-PointTwo.y)
		+(double)(PointOne.z-PointTwo.z)*(double)(PointOne.z-PointTwo.z));
}

//---------------------------------------------------------------------
// BOOL IntersectTriangle( const Triangle &triangle, const LINE &Line, D3DXVECTOR3 &intersectPnt )
// 
// Usage: Finds the intersection between a line between two vertices (v1 and v2) with a 3D triangle
//
// Return true if the line intersection with the triangle
//
// Note, 
// 1)this is the basis of all non spherical collision detection, it is borrowed from the directX
//   SDK with some modifications from Tom.
// 2) This only checks one face of the triangle, as a result this function has to be called twice
//---------------------------------------------------------------------
BOOL UMath::IntersectTriangle(const Triangle &triangle, const LINE &Line, D3DXVECTOR3 &intersectPnt)
{
	    // Find vectors for two edges sharing vert0
    D3DXVECTOR3 edge1 = triangle.v1 - triangle.v0;
    D3DXVECTOR3 edge2 = triangle.v2 - triangle.v0;

    // Begin calculating determinant

	// calculate in the form of ray direction not line end point as it sent over.
	D3DXVECTOR3 tDirect = Line.Point2 - Line.Point1;

    D3DXVECTOR3 pvec;
    D3DXVec3Cross( &pvec, &tDirect/*&ray.direction*/, &edge2 );

    // If determinant is near zero, ray lies in plane of triangle
    FLOAT det = D3DXVec3Dot( &edge1, &pvec );
    if( det < 0.0001f ) return FALSE;

    // Calculate distance from vert0 to ray origin
    D3DXVECTOR3 tvec = Line.Point1 - triangle.v0;

    // Calculate U parameter and test bounds
    float u = D3DXVec3Dot( &tvec, &pvec );
    if( u < 0.0f || u > det ) return FALSE;

    // Prepare to test V parameter
    D3DXVECTOR3 qvec;
    D3DXVec3Cross( &qvec, &tvec, &edge1 );

    // Calculate V parameter and test bounds
    float v = D3DXVec3Dot( &tDirect, &qvec );
    if( v < 0.0f || u + v > det ) return FALSE;

	// The ray intersects the triangle, however the line might not, so check to see if the
	// line intersects the triangle, this is a mod from the original code that is used for 
	// ray tracing not collision detection
	PLANE Plane;

	// carefull with the fourth arguament but I'm reusing code that the fourth argument doesn't matter
	Plane = PLANE(triangle.v0,triangle.v1,triangle.v2,triangle.v0);
	GeneratePlane(Plane);

	// lazy, dont' want to change the const ray
	LINE tLine = Line;

	// return true if the line intersects the plane, false otherwise
	if (FindIntersection(intersectPnt,Plane,tLine.Point1,tLine.Point2))
		return TRUE;

	return FALSE;

}

//======================================================================
// float Gaussian(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Standard gaussian function of A*exp(-D*D/2D0*D0)
// The D0 is passed as Args[0], D is Distance
// A, the height constant is ARgs[1], 
//
// Usage: This function is used for 3D height calculates.
//======================================================================
float Gaussian(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	// gets the distance
    float Distance = (float)sqrt((Center.x - Current.x)*(Center.x - Current.x) + 
			(Center.z - Current.z)*(Center.z - Current.z));
	float Variable;
	Variable = -Distance*Distance/(2*Args[0]*Args[0]);
	return (float)(Args[1]*exp(Variable));
}

//======================================================================
// float Cos3Da(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
//
// Returns a 3D cos height paramter (the cos of the Radius of Dx, Dz)
//======================================================================
float Cos3Da(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	// gets the distance
  	float D = (float)sqrt((double)(Center.x-Current.x)*(double)(Center.x-Current.x) + (double)(Center.z-Current.z)*(double)(Center.z-Current.z));

	if (D > Args[0])
		return EXITCODE;

	// multiply by PI so 1 period is 1 period (actually half of a period)
	return Args[1]*(float)cos(PI*D/Args[0]);
}

//======================================================================
// float Cos3Db(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Returns a 3D cos height paramter (the multiplication of two cos of Dx and Dz)
//======================================================================
float Cos3Db(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
//	float D = (float)sqrt((double)(Center.x-Current.x)*(double)(Center.x-Current.x) + (double)(Center.z-Current.z)*(double)(Center.z-Current.z));

	float Dx = (Center.x-Current.x);
	if (Dx < 0)
		Dx = -Dx;
	if (Dx > Args[0])
		return EXITCODE;
	float Dz = (Center.z - Current.z);
	if (Dz < 0)
		Dz = -Dz;
	if (Dz > Args[0])
		return EXITCODE;


	// gets the distance
    float H1 = (float)cos(PI*(Dx)/Args[0]);
	float H2 = (float)cos(PI*(Dz)/Args[0]);

	return Args[1]*(H1*H2);
}

//======================================================================
// float Cos3Dc(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Returns a 3D cos height paramter (the addition of two cos of Dx and Dz)
//======================================================================
float Cos3Dc(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float Dx = (Center.x-Current.x);
	if (Dx < 0)
		Dx = -Dx;
	if (Dx > Args[0])
		return EXITCODE;
	float Dz = (Center.z - Current.z);
	if (Dz < 0)
		Dz = -Dz;
	if (Dz > Args[0])
		return EXITCODE;


	// gets the distance
    float H1 = (float)cos(PI*(Dx)/Args[0]);
	float H2 = (float)cos(PI*(Dz)/Args[0]);

	return Args[1]*(H1+H2)/2;
}

//======================================================================
// float Cos3DRa(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D cos height paramter (the cos of the Radius of Dx, Dz, that has a ending radius)
//======================================================================
float Cos3DRa(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float D = (float)sqrt((double)(Center.x-Current.x)*(double)(Center.x-Current.x) + (double)(Center.z-Current.z)*(double)(Center.z-Current.z));

	if (D >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// multiply by PI so 1 period is 1 period (actually half of a period)
	return Args[1]*(float)cos(PI*D/Args[0]);
}

//======================================================================
// float Cos3DRa(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D cos height paramter 
// (the cos of the Radius of Dx, Dz, that has a ending linear decreasing radius)
//======================================================================
float Cos3DLDRa(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float D = (float)sqrt((double)(Center.x-Current.x)*(double)(Center.x-Current.x) + (double)(Center.z-Current.z)*(double)(Center.z-Current.z));

	if (D >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// multiply by PI so 1 period is 1 period (actually half of a period)
	return (Args[1]*(float)cos(PI*D/Args[0]))*(1-D/Args[2]);
}

//======================================================================
// float Cos3DRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D cos height paramter
// (the multiplication of two cos of Dx and Dz with an ending radius)
//======================================================================
float Cos3DRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float Dx = (Center.x-Current.x);
	if (Dx < 0)
		Dx = -Dx;
	if (Dx >= Args[2] || Args[0] == 0)
		return EXITCODE;
	float Dz = (Center.z - Current.z);
	if (Dz < 0)
		Dz = -Dz;
	if (Dz >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// gets the distance
    float H1 = (float)cos(PI*(Dx)/Args[0]);
	float H2 = (float)cos(PI*(Dz)/Args[0]);


	return Args[1]*(H1*H2);

}

//======================================================================
// float Cos3DRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D cos height paramter
// // (the multiplication of two cos of Dx and Dz with an linear decreasing ending radius)
//======================================================================
float Cos3DLDRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float Dx = (Center.x-Current.x);
	float Dz = (Center.z - Current.z);

	float D = (float)sqrt(Dx*Dx + Dz*Dz);
	if (D >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// gets the distance
    float H1 = (float)cos(PI*(Dx)/Args[0]);
	float H2 = (float)cos(PI*(Dz)/Args[0]);

	return Args[1]*(H1*H2)*(1-D/Args[2]);
}

//======================================================================
// float Cos3DRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D cos height paramter
// (the multiplication of two cos of Dx and Dz with an ending radius)
//======================================================================
float Cos3DRc(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float Dx = (Center.x-Current.x);
	if (Dx < 0)
		Dx = -Dx;
	if (Dx >= Args[2] || Args[0] == 0)
		return EXITCODE;
	float Dz = (Center.z - Current.z);
	if (Dz < 0)
		Dz = -Dz;
	if (Dz >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// gets the distance
    float H1 = (float)cos(PI*(Dx)/Args[0]);
	float H2 = (float)cos(PI*(Dz)/Args[0]);

	return Args[1]*(H1+H2);

}

//======================================================================
// float Cos3DRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D cos height paramter
// // (the multiplication of two cos of Dx and Dz with an linear decreasing ending radius)
//======================================================================
float Cos3DLDRc(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float Dx = (Center.x-Current.x);
	float Dz = (Center.z - Current.z);

	float D = (float)sqrt(Dx*Dx + Dz*Dz);
	if (D >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// gets the distance
    float H1 = (float)cos(PI*(Dx)/Args[0]);
	float H2 = (float)cos(PI*(Dz)/Args[0]);

	return Args[1]*(H1+H2)*(1-D/Args[2]);
}

//======================================================================
// float Sin3Da(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
//
// Returns a 3D Sin height paramter (the Sin of the Radius of Dx, Dz)
//======================================================================
float Sin3Da(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	// gets the distance
  	float D = (float)sqrt((double)(Center.x-Current.x)*(double)(Center.x-Current.x) + (double)(Center.z-Current.z)*(double)(Center.z-Current.z));

	if (D > Args[0])
		return EXITCODE;

	// multiply by PI so 1 period is 1 period (actually half of a period)
	return Args[1]*(float)sin(PI*D/Args[0]);
}

//======================================================================
// float Sin3Db(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Returns a 3D Sin height paramter (the multiplication of two Sin of Dx and Dz)
//======================================================================
float Sin3Db(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
//	float D = (float)sqrt((double)(Center.x-Current.x)*(double)(Center.x-Current.x) + (double)(Center.z-Current.z)*(double)(Center.z-Current.z));

	float Dx = (Center.x-Current.x);
	if (Dx < 0)
		Dx = -Dx;
	if (Dx > Args[0])
		return EXITCODE;
	float Dz = (Center.z - Current.z);
	if (Dz < 0)
		Dz = -Dz;
	if (Dz > Args[0])
		return EXITCODE;


	// gets the distance
    float H1 = (float)sin(PI*(Dx)/Args[0]);
	float H2 = (float)sin(PI*(Dz)/Args[0]);

	return Args[1]*(H1*H2);
}

//======================================================================
// float Sin3Dc(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Returns a 3D Sin height paramter (the addition of two Sin of Dx and Dz)
//======================================================================
float Sin3Dc(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float Dx = (Center.x-Current.x);
	if (Dx < 0)
		Dx = -Dx;
	if (Dx > Args[0])
		return EXITCODE;
	float Dz = (Center.z - Current.z);
	if (Dz < 0)
		Dz = -Dz;
	if (Dz > Args[0])
		return EXITCODE;

	// gets the distance
    float H1 = (float)sin(PI*(Dx)/Args[0]);
	float H2 = (float)sin(PI*(Dz)/Args[0]);

	return Args[1]*(H1+H2)/2;
}

//======================================================================
// float Sin3DRa(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D Sin height paramter (the Sin of the Radius of Dx, Dz, that has a ending radius)
//======================================================================
float Sin3DRa(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float D = (float)sqrt((double)(Center.x-Current.x)*(double)(Center.x-Current.x) + (double)(Center.z-Current.z)*(double)(Center.z-Current.z));

	if (D >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// multiply by PI so 1 period is 1 period (actually half of a period)
	return Args[1]*(float)sin(PI*D/Args[0]);
}

//======================================================================
// float Sin3DRa(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D Sin height paramter 
// (the Sin of the Radius of Dx, Dz, that has a ending linear decreasing radius)
//======================================================================
float Sin3DLDRa(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float D = (float)sqrt((double)(Center.x-Current.x)*(double)(Center.x-Current.x) + (double)(Center.z-Current.z)*(double)(Center.z-Current.z));

	if (D >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// multiply by PI so 1 period is 1 period (actually half of a period)
	return (Args[1]*(float)sin(PI*D/Args[0]))*(1-D/Args[2]);
}

//======================================================================
// float Sin3DRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D Sin height paramter
// (the multiplication of two Sin of Dx and Dz with an ending radius)
//======================================================================
float Sin3DRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float Dx = (Center.x-Current.x);
	if (Dx < 0)
		Dx = -Dx;
	if (Dx >= Args[2] || Args[0] == 0)
		return EXITCODE;
	float Dz = (Center.z - Current.z);
	if (Dz < 0)
		Dz = -Dz;
	if (Dz >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// gets the distance
    float H1 = (float)sin(PI*(Dx)/Args[0]);
	float H2 = (float)sin(PI*(Dz)/Args[0]);

	return Args[1]*(H1*H2);

}

//======================================================================
// float Sin3DRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D Sin height paramter
// // (the multiplication of two Sin of Dx and Dz with an linear decreasing ending radius)
//======================================================================
float Sin3DLDRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float Dx = (Center.x-Current.x);
	float Dz = (Center.z - Current.z);

	float D = (float)sqrt(Dx*Dx + Dz*Dz);
	if (D >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// gets the distance
    float H1 = (float)sin(PI*(Dx)/Args[0]);
	float H2 = (float)sin(PI*(Dz)/Args[0]);

	return Args[1]*(H1*H2)*(1-D/Args[2]);
}

//======================================================================
// float Sin3DRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D Sin height paramter
// (the multiplication of two Sin of Dx and Dz with an ending radius)
//======================================================================
float Sin3DRc(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float Dx = (Center.x-Current.x);
	if (Dx < 0)
		Dx = -Dx;
	if (Dx >= Args[2] || Args[0] == 0)
		return EXITCODE;
	float Dz = (Center.z - Current.z);
	if (Dz < 0)
		Dz = -Dz;
	if (Dz >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// gets the distance
    float H1 = (float)sin(PI*(Dx)/Args[0]);
	float H2 = (float)sin(PI*(Dz)/Args[0]);

	return Args[1]*(H1+H2);

}

//======================================================================
// float Sin3DRb(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
//
// Args[0] is the phase length
// Args[1] is the height constant
// Args[2] is the maximum radius
// Returns a 3D Sin height paramter
// // (the multiplication of two Sin of Dx and Dz with an linear decreasing ending radius)
//======================================================================
float Sin3DLDRc(const D3DXVECTOR3 &Center, const D3DXVECTOR3 &Current, const float Args[])
{
	float Dx = (Center.x-Current.x);
	float Dz = (Center.z - Current.z);

	float D = (float)sqrt(Dx*Dx + Dz*Dz);
	if (D >= Args[2] || Args[0] == 0)
		return EXITCODE;

	// gets the distance
    float H1 = (float)sin(PI*(Dx)/Args[0]);
	float H2 = (float)sin(PI*(Dz)/Args[0]);

	return Args[1]*(H1+H2)*(1-D/Args[2]);
}


